package com.vungle.ads

import android.os.Build
import androidx.annotation.VisibleForTesting
import androidx.annotation.WorkerThread
import com.vungle.ads.internal.executor.VungleThreadPoolExecutor
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.network.VungleHeader
import com.vungle.ads.internal.platform.Platform.Companion.MANUFACTURER_AMAZON
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.util.ActivityManager
import com.vungle.ads.internal.util.LogEntry
import com.vungle.ads.internal.util.Logger
import java.util.concurrent.BlockingQueue
import java.util.concurrent.Executors
import java.util.concurrent.LinkedBlockingQueue
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicBoolean

/**
 * Main Class for tracking analytics
 */

object AnalyticsClient {

    enum class LogLevel(val level: Int) {
        ERROR_LOG_LEVEL_OFF(0), ERROR_LOG_LEVEL_ERROR(1), ERROR_LOG_LEVEL_DEBUG(2);

        companion object {
            fun fromValue(logLevel: Int): LogLevel {
                when (logLevel) {
                    ERROR_LOG_LEVEL_DEBUG.level -> {
                        return ERROR_LOG_LEVEL_DEBUG
                    }

                    ERROR_LOG_LEVEL_ERROR.level -> {
                        return ERROR_LOG_LEVEL_ERROR
                    }

                    ERROR_LOG_LEVEL_OFF.level -> {
                        return ERROR_LOG_LEVEL_OFF
                    }
                }
                return ERROR_LOG_LEVEL_ERROR
            }
        }
    }

    private const val TAG: String = "AnalyticsClient"

    @VisibleForTesting
    internal val errors: BlockingQueue<Sdk.SDKError.Builder> =
        LinkedBlockingQueue()

    @VisibleForTesting
    internal val metrics: BlockingQueue<Sdk.SDKMetric.Builder> =
        LinkedBlockingQueue()

    @VisibleForTesting
    internal val pendingErrors: BlockingQueue<Sdk.SDKError.Builder> =
        LinkedBlockingQueue()

    @VisibleForTesting
    internal val pendingMetrics: BlockingQueue<Sdk.SDKMetric.Builder> =
        LinkedBlockingQueue()

    /**
     * Max time before any queued items are sent out
     */
    private const val REFRESH_TIME_MILLIS = 5000L

    /**
     * Max number of items queued before we send them out
     */
    private const val MAX_BATCH_SIZE = 20

    @VisibleForTesting
    internal var vungleApiClient: VungleApiClient? = null

    @VisibleForTesting
    internal var executor: VungleThreadPoolExecutor? = null

    @VisibleForTesting
    internal var metricsEnabled = false

    private var logLevel = LogLevel.ERROR_LOG_LEVEL_ERROR

    @VisibleForTesting
    internal var refreshEnabled = true

    @VisibleForTesting
    internal val isInitialized = AtomicBoolean(false)

    @Synchronized
    internal fun initOrUpdate(
        vungleApiClient: VungleApiClient,
        executor: VungleThreadPoolExecutor,
        errorLogLevel: Int,
        metricsEnabled: Boolean,
    ) {

        this.logLevel = LogLevel.fromValue(errorLogLevel)
        this.metricsEnabled = metricsEnabled

        when (errorLogLevel) {
            LogLevel.ERROR_LOG_LEVEL_DEBUG.level -> {
                Logger.enable(true)
            }

            LogLevel.ERROR_LOG_LEVEL_ERROR.level -> {
                Logger.enable(false)
            }

            LogLevel.ERROR_LOG_LEVEL_OFF.level -> {
                Logger.enable(false)
            }
        }

        if (isInitialized.getAndSet(true)) {
            Logger.d(TAG, "AnalyticsClient already initialized")
            return
        }

        this.executor = executor
        this.vungleApiClient = vungleApiClient

        try {
            if (pendingErrors.isNotEmpty()) {
                pendingErrors.drainTo(errors)
            }
        } catch (ex: Exception) {
            Logger.e(TAG, "Failed to add pendingErrors to errors queue.", ex)
        }
        try {
            if (pendingMetrics.isNotEmpty()) {
                pendingMetrics.drainTo(metrics)
            }
        } catch (ex: Exception) {
            Logger.e(TAG, "Failed to add pendingMetrics to metrics queue.", ex)
        }

        if (refreshEnabled) {
            Executors.newSingleThreadScheduledExecutor()
                .scheduleWithFixedDelay(
                    {
                        // Execute all tasks in the same thread
                        executor.execute {
                            report()
                        }
                    },
                    5000, // Start after 5 seconds
                    REFRESH_TIME_MILLIS,
                    TimeUnit.MILLISECONDS
                )
        }

    }

    @Synchronized
    internal fun logError(
        reason: Sdk.SDKError.Reason,
        message: String,
        entry: LogEntry? = null,
    ) {
        try {
            if (executor == null) {
                val error = genSDKError(reason, message, entry)
                pendingErrors.put(error)
                return
            }

            executor?.execute {
                logErrorInSameThread(reason, message, entry)
            }
        } catch (ex: Exception) {
            Logger.e(TAG, "Cannot logError $reason, $message, $entry", ex)
        }
    }

    private fun genSDKError(
        reason: Sdk.SDKError.Reason,
        message: String,
        entry: LogEntry? = null,
    ): Sdk.SDKError.Builder {
        return Sdk.SDKError.newBuilder()
            .setOs(if (MANUFACTURER_AMAZON == Build.MANUFACTURER) "amazon" else "android")
            .setOsVersion(Build.VERSION.SDK_INT.toString())
            .setMake(Build.MANUFACTURER)
            .setModel(Build.MODEL)
            .setReason(reason)
            .setMessage(message)
            .setAt(System.currentTimeMillis())
            .setPlacementReferenceId(entry?.placementRefId ?: "")
            .setCreativeId(entry?.creativeId ?: "")
            .setEventId(entry?.eventId ?: "")
            .setAdSource(entry?.adSource ?: "")
            .setVmVersion(entry?.vmVersion ?: "")
            .setMediationName(entry?.mediationName ?: VungleHeader.headerUa)
            .setAppState(if (ActivityManager.isForeground()) 0 else 2)

    }

    @Synchronized
    private fun logErrorInSameThread(
        reason: Sdk.SDKError.Reason,
        message: String,
        entry: LogEntry? = null,
    ) {
        if (logLevel == LogLevel.ERROR_LOG_LEVEL_OFF) {
            return
        }
        try {
            val error = genSDKError(reason, message, entry)

            this.errors.put(error)
            Logger.w(TAG, "Logging error: $reason with message: $message, mediation: ${error.mediationName}")
            if (errors.size >= MAX_BATCH_SIZE) {
                report()
            }
        } catch (ex: Exception) {
            Logger.e(TAG, "Cannot logError", ex)
        }

    }

    @Synchronized
    internal fun logMetric(
        metricType: Sdk.SDKMetric.SDKMetricType,
        metricValue: Long = 0,
        logEntry: LogEntry? = null,
        metaData: String? = null,
    ) {
        try {
            if (executor == null) {
                val metric =
                    genMetric(metricType, metricValue, logEntry, metaData)
                pendingMetrics.put(metric)
                return
            }

            executor?.execute {
                logMetricInSameThread(
                    metricType,
                    metricValue,
                    logEntry,
                    metaData
                )
            }
        } catch (ex: Exception) {
            Logger.e(
                TAG, "Cannot logMetric $metricType, $metricValue, $logEntry, $metaData", ex
            )
        }
    }

    private fun genMetric(
        metricType: Sdk.SDKMetric.SDKMetricType,
        metricValue: Long = 0,
        logEntry: LogEntry? = null,
        metaData: String? = null,
    ): Sdk.SDKMetric.Builder {
        return Sdk.SDKMetric.newBuilder()
            .setType(metricType)
            .setValue(metricValue)
            .setMake(Build.MANUFACTURER)
            .setModel(Build.MODEL)
            .setOs(if (MANUFACTURER_AMAZON == Build.MANUFACTURER) "amazon" else "android")
            .setOsVersion(Build.VERSION.SDK_INT.toString())
            .setPlacementReferenceId(logEntry?.placementRefId ?: "")
            .setCreativeId(logEntry?.creativeId ?: "")
            .setEventId(logEntry?.eventId ?: "")
            .setMeta(metaData ?: "")
            .setMediationName(logEntry?.mediationName ?: VungleHeader.headerUa)
            .setAdSource(logEntry?.adSource ?: "")
            .setVmVersion(logEntry?.vmVersion ?: "")
            .setAppState(if (ActivityManager.isForeground()) 0 else 2)
    }

    @Synchronized
    private fun logMetricInSameThread(
        metricType: Sdk.SDKMetric.SDKMetricType,
        metricValue: Long = 0,
        logEntry: LogEntry? = null,
        metaData: String? = null,
    ) {
        if (!metricsEnabled) {
            return
        }
        try {
            val metric =
                genMetric(metricType, metricValue, logEntry, metaData)

            this.metrics.put(metric)
            Logger.w(
                TAG,
                "Logging Metric $metricType with value $metricValue for placement ${logEntry?.placementRefId} mediation:${metric.mediationName}"
            )
            if (metrics.size >= MAX_BATCH_SIZE) {
                report()
            }
        } catch (ex: Exception) {
            Logger.e(TAG, "Cannot logMetrics", ex)
        }
    }

    @Synchronized
    internal fun logMetric(
        singleValueMetric: SingleValueMetric,
        logEntry: LogEntry? = null,
        metaData: String? = singleValueMetric.meta,
    ) {
        logMetric(
            singleValueMetric.metricType,
            singleValueMetric.getValue(),
            logEntry,
            metaData
        )
    }

    @Synchronized
    internal fun logMetric(
        timeIntervalMetric: TimeIntervalMetric,
        logEntry: LogEntry? = null,
        metaData: String? = timeIntervalMetric.meta,
    ) {
        logMetric(
            timeIntervalMetric.metricType,
            timeIntervalMetric.getValue(),
            logEntry,
            metaData
        )
    }

    @Synchronized
    internal fun logMetric(
        oneShotTimeIntervalMetric: OneShotTimeIntervalMetric,
        logEntry: LogEntry? = null,
        metaData: String? = oneShotTimeIntervalMetric.meta,
    ) {
        if (!oneShotTimeIntervalMetric.isLogged()) {
            logMetric(
                oneShotTimeIntervalMetric as TimeIntervalMetric,
                logEntry,
                metaData
            )
            oneShotTimeIntervalMetric.markLogged()
        }
    }

    @Synchronized
    private fun report() {
        if (logLevel != LogLevel.ERROR_LOG_LEVEL_OFF && errors.size > 0) {
            flushErrors()
        }

        if (metricsEnabled && metrics.size > 0) {
            flushMetrics()
        }
    }

    @WorkerThread
    private fun flushMetrics() {
        Logger.d(TAG, message = "Sending ${this.metrics.size} metrics")
        val currentSendingMetrics: BlockingQueue<Sdk.SDKMetric.Builder> =
            LinkedBlockingQueue()
        metrics.drainTo(currentSendingMetrics)
        if (currentSendingMetrics.isEmpty()) {/* There is a somewhat slim chance that upon getting
             here there is nothing to send. And we don't need those empty calls to server */
            return
        }
        vungleApiClient?.reportMetrics(currentSendingMetrics, object : RequestListener {
            override fun onSuccess() {
                Logger.d(TAG, message = "Sent ${currentSendingMetrics.size} metrics")
            }

            override fun onFailure() {
                Logger.d(
                    TAG,
                    message = "Failed to send ${currentSendingMetrics.size} metrics"
                )
                metrics.addAll(currentSendingMetrics)
            }

        })
    }

    @WorkerThread
    private fun flushErrors() {
        Logger.d(TAG, message = "Sending ${this.errors.size} errors")
        val currentSendingErrors: BlockingQueue<Sdk.SDKError.Builder> =
            LinkedBlockingQueue()
        errors.drainTo(currentSendingErrors)
        if (currentSendingErrors.isEmpty()) {/* There is a somewhat slim chance that upon getting
             here there is nothing to send. And we don't need those empty calls to server */
            return
        }
        vungleApiClient?.reportErrors(currentSendingErrors, object : RequestListener {
            override fun onSuccess() {
                Logger.d(TAG, message = "Sent ${currentSendingErrors.size} errors")
            }

            override fun onFailure() {
                Logger.d(
                    TAG,
                    message = "Failed to send ${currentSendingErrors.size} errors"
                )
                errors.addAll(currentSendingErrors)
            }
        })
    }

    interface RequestListener {
        fun onSuccess()
        fun onFailure()
    }

}
