/*
 * Decompiled with CFR 0.152.
 */
package com.vmware.xenon.common;

import com.vmware.xenon.common.FactoryService;
import com.vmware.xenon.common.Operation;
import com.vmware.xenon.common.OperationContext;
import com.vmware.xenon.common.Service;
import com.vmware.xenon.common.ServiceDocument;
import com.vmware.xenon.common.ServiceHost;
import com.vmware.xenon.common.ServiceStats;
import com.vmware.xenon.common.SystemHostInfo;
import com.vmware.xenon.common.UriUtils;
import com.vmware.xenon.common.UtilityService;
import com.vmware.xenon.common.Utils;
import com.vmware.xenon.services.common.ServiceContextIndexService;
import java.lang.management.ManagementFactory;
import java.lang.management.ThreadMXBean;
import java.util.EnumSet;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

class ServiceResourceTracker {
    private final Map<String, Service> attachedServices;
    private final Map<String, Service> pendingPauseServices;
    private final ConcurrentSkipListSet<String> serviceFactoriesUnderMemoryPressure = new ConcurrentSkipListSet();
    private final ConcurrentMap<String, ServiceDocument> cachedServiceStates = new ConcurrentHashMap<String, ServiceDocument>();
    private final ConcurrentMap<String, Long> persistedServiceLastAccessTimes = new ConcurrentHashMap<String, Long>();
    private final ConcurrentMap<CachedServiceStateKey, ServiceDocument> cachedTransactionalServiceStates = new ConcurrentHashMap<CachedServiceStateKey, ServiceDocument>();
    private final ServiceHost host;
    private boolean isServiceStateCaching = true;
    private long startTimeMicros;

    public static ServiceResourceTracker create(ServiceHost host, Map<String, Service> services, Map<String, Service> pendingPauseServices) {
        ServiceResourceTracker srt = new ServiceResourceTracker(host, services, pendingPauseServices);
        return srt;
    }

    public ServiceResourceTracker(ServiceHost host, Map<String, Service> services, Map<String, Service> pendingPauseServices) {
        this.attachedServices = services;
        this.pendingPauseServices = pendingPauseServices;
        this.host = host;
    }

    private void checkAndInitializeStats() {
        if (this.startTimeMicros > 0L) {
            return;
        }
        this.startTimeMicros = Utils.getNowMicrosUtc();
        if (this.host.getManagementService() == null) {
            this.host.log(Level.WARNING, "Management service not found, stats will not be available", new Object[0]);
            return;
        }
        long inUseMem = this.host.getState().systemInfo.totalMemoryByteCount - this.host.getState().systemInfo.freeMemoryByteCount;
        long freeMem = this.host.getState().systemInfo.maxMemoryByteCount - inUseMem;
        long freeDisk = this.host.getState().systemInfo.freeDiskByteCount;
        this.createTimeSeriesStat("availableMemoryBytes", freeMem);
        this.createTimeSeriesStat("availableDiskByte", freeDisk);
        this.createTimeSeriesStat("cpuUsagePercent", 0.0);
        this.createTimeSeriesStat("threadCount", Utils.DEFAULT_THREAD_COUNT);
    }

    private void createTimeSeriesStat(String name, double v) {
        this.createDayTimeSeriesStat(name, v);
        this.createHourTimeSeriesStat(name, v);
    }

    private void createDayTimeSeriesStat(String name, double v) {
        Service mgmtService = this.host.getManagementService();
        ServiceStats.ServiceStat st = new ServiceStats.ServiceStat();
        st.name = name + "PerDay";
        st.timeSeriesStats = new ServiceStats.TimeSeriesStats((int)TimeUnit.DAYS.toHours(1L), TimeUnit.HOURS.toMillis(1L), EnumSet.of(ServiceStats.TimeSeriesStats.AggregationType.AVG));
        mgmtService.setStat(st, v);
    }

    private void createHourTimeSeriesStat(String name, double v) {
        Service mgmtService = this.host.getManagementService();
        ServiceStats.ServiceStat st = new ServiceStats.ServiceStat();
        st.name = name + "PerHour";
        st.timeSeriesStats = new ServiceStats.TimeSeriesStats((int)TimeUnit.HOURS.toMinutes(1L), TimeUnit.MINUTES.toMillis(1L), EnumSet.of(ServiceStats.TimeSeriesStats.AggregationType.AVG));
        mgmtService.setStat(st, v);
    }

    private void updateStats(long now) {
        long[] threadIds;
        SystemHostInfo shi = this.host.updateSystemInfo(false);
        Service mgmtService = this.host.getManagementService();
        this.checkAndInitializeStats();
        long freeMemory = shi.maxMemoryByteCount - (shi.totalMemoryByteCount - shi.freeMemoryByteCount);
        mgmtService.setStat("availableMemoryBytesPerHour", (double)freeMemory);
        mgmtService.setStat("availableMemoryBytesPerDay", (double)freeMemory);
        mgmtService.setStat("availableDiskBytePerHour", (double)shi.freeDiskByteCount);
        mgmtService.setStat("availableDiskBytePerDay", (double)shi.freeDiskByteCount);
        ThreadMXBean threadBean = ManagementFactory.getThreadMXBean();
        if (!threadBean.isCurrentThreadCpuTimeSupported()) {
            return;
        }
        long totalTime = 0L;
        for (long threadId : threadIds = threadBean.getAllThreadIds()) {
            totalTime += threadBean.getThreadCpuTime(threadId);
        }
        double runningTime = now - this.startTimeMicros;
        if (runningTime <= 0.0) {
            return;
        }
        mgmtService.setStat("threadCountPerDay", (double)threadIds.length);
        mgmtService.setStat("threadCountPerHour", (double)threadIds.length);
        totalTime = TimeUnit.NANOSECONDS.toMicros(totalTime);
        double pctUse = (double)totalTime / runningTime;
        mgmtService.setStat("cpuUsagePercentPerHour", pctUse);
        mgmtService.setStat("cpuUsagePercentPerDay", pctUse);
    }

    public void setServiceStateCaching(boolean enable) {
        this.isServiceStateCaching = enable;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateCachedServiceState(Service s, ServiceDocument st, Operation op) {
        if (ServiceHost.isServiceIndexed(s) && !this.isTransactional(op)) {
            this.persistedServiceLastAccessTimes.put(s.getSelfLink(), Utils.getNowMicrosUtc());
        }
        if (ServiceHost.isServiceIndexed(s) && !this.isServiceStateCaching) {
            return;
        }
        if (!this.isTransactional(op)) {
            String string = s.getSelfLink();
            synchronized (string) {
                ServiceDocument cachedState = this.cachedServiceStates.put(s.getSelfLink(), st);
                if (cachedState != null && cachedState.documentVersion > st.documentVersion) {
                    this.cachedServiceStates.put(s.getSelfLink(), cachedState);
                }
            }
            return;
        }
        CachedServiceStateKey key = new CachedServiceStateKey(s.getSelfLink(), op.getTransactionId());
        String string = key.toString();
        synchronized (string) {
            ServiceDocument cachedState = this.cachedTransactionalServiceStates.put(key, st);
            if (cachedState != null && cachedState.documentVersion > st.documentVersion) {
                this.cachedTransactionalServiceStates.put(key, cachedState);
            }
        }
    }

    public ServiceDocument getCachedServiceState(Service s, Operation op) {
        String servicePath = s.getSelfLink();
        ServiceDocument state = null;
        if (this.isTransactional(op)) {
            CachedServiceStateKey key = new CachedServiceStateKey(servicePath, op.getTransactionId());
            state = (ServiceDocument)this.cachedTransactionalServiceStates.get(key);
        } else if (ServiceHost.isServiceIndexed(s)) {
            this.persistedServiceLastAccessTimes.put(servicePath, this.host.getStateNoCloning().lastMaintenanceTimeUtcMicros);
        }
        if (state == null) {
            state = (ServiceDocument)this.cachedServiceStates.get(servicePath);
        }
        if (state == null) {
            return null;
        }
        if (state.documentExpirationTimeMicros > 0L && state.documentExpirationTimeMicros < state.documentUpdateTimeMicros) {
            this.stopService(servicePath, true, op);
            return null;
        }
        return state;
    }

    private void stopService(String servicePath, boolean isExpired, Operation op) {
        Service s = this.host.findService(servicePath, true);
        if (s == null) {
            return;
        }
        if (isExpired && s.hasOption(Service.ServiceOption.PERSISTENCE)) {
            return;
        }
        Operation deleteExp = Operation.createDelete(s.getUri()).disableFailureLogging(true).addPragmaDirective("xn-no-index-update").setReplicationDisabled(true).setReferer(this.host.getUri()).setCompletion((o, e) -> {
            if (s.hasOption(Service.ServiceOption.ON_DEMAND_LOAD)) {
                this.host.getManagementService().adjustStat("onDemandLoadStopCount", 1.0);
            }
        });
        this.host.sendRequest(deleteExp);
        this.clearCachedServiceState(servicePath, op);
    }

    public void clearCachedServiceState(String servicePath, Operation op) {
        this.clearCachedServiceState(servicePath, op, false);
    }

    private void clearCachedServiceState(String servicePath, Operation op, boolean keepLastAccessTime) {
        if (!this.isTransactional(op)) {
            if (!keepLastAccessTime) {
                this.persistedServiceLastAccessTimes.remove(servicePath);
            }
            ServiceDocument doc = (ServiceDocument)this.cachedServiceStates.remove(servicePath);
            Service s = this.host.findService(servicePath, true);
            if (s == null) {
                return;
            }
            if (doc != null) {
                this.updateCacheClearStats(s);
            }
            return;
        }
        this.clearTransactionalCachedServiceState(servicePath, op.getTransactionId());
    }

    public void clearTransactionalCachedServiceState(String servicePath, String transactionId) {
        CachedServiceStateKey key = new CachedServiceStateKey(servicePath, transactionId);
        ServiceDocument doc = (ServiceDocument)this.cachedTransactionalServiceStates.remove(key);
        Service s = this.host.findService(servicePath, true);
        if (s == null) {
            return;
        }
        if (doc != null) {
            this.updateCacheClearStats(s);
        }
    }

    private void updateCacheClearStats(Service s) {
        s.adjustStat("stateCacheClearCount", 1.0);
        this.host.getManagementService().adjustStat("serviceCacheClearCount", 1.0);
        if (s.hasOption(Service.ServiceOption.ON_DEMAND_LOAD)) {
            this.host.getManagementService().adjustStat("onDemandLoadCacheClearCount", 1.0);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void performMaintenance(long now, long deadlineMicros) {
        this.updateStats(now);
        ServiceHost.ServiceHostState hostState = this.host.getStateNoCloning();
        long memoryLimitLowMB = this.host.getServiceMemoryLimitMB("", ServiceHost.ServiceHostState.MemoryLimitType.HIGH_WATERMARK);
        long memoryInUseMB = hostState.serviceCount * 4096L;
        boolean shouldPause = memoryLimitLowMB <= (memoryInUseMB /= 0x100000L);
        int pauseServiceCount = 0;
        for (Service service : this.attachedServices.values()) {
            String factoryPath;
            Service existing;
            boolean cacheCleared;
            if (service.hasOption(Service.ServiceOption.FACTORY)) continue;
            ServiceDocument s = (ServiceDocument)this.cachedServiceStates.get(service.getSelfLink());
            Long lastAccessTime = (Long)this.persistedServiceLastAccessTimes.get(service.getSelfLink());
            boolean bl = cacheCleared = s == null;
            if (s != null) {
                if (!ServiceHost.isServiceIndexed(service)) {
                    if (s.documentExpirationTimeMicros <= 0L || s.documentExpirationTimeMicros >= now) continue;
                    this.stopService(service.getSelfLink(), true, null);
                    continue;
                }
                if (service.hasOption(Service.ServiceOption.TRANSACTION_PENDING)) continue;
                if (lastAccessTime == null) {
                    lastAccessTime = s.documentUpdateTimeMicros;
                }
                if (hostState.serviceCacheClearDelayMicros + lastAccessTime < now) {
                    this.clearCachedServiceState(service.getSelfLink(), null, true);
                    cacheCleared = true;
                }
            }
            if (lastAccessTime == null && ServiceHost.isServiceIndexed(service) || lastAccessTime != null && hostState.lastMaintenanceTimeUtcMicros - lastAccessTime < service.getMaintenanceIntervalMicros() * 2L || service.hasOption(Service.ServiceOption.PERIODIC_MAINTENANCE) || !service.hasOption(Service.ServiceOption.FACTORY_ITEM) || this.host.isServiceStarting(service, service.getSelfLink())) continue;
            if (this.host.hasPendingServiceAvailableCompletions(service.getSelfLink())) {
                this.host.log(Level.INFO, "Pending available completions, skipping pause/stop on %s", service.getSelfLink());
                continue;
            }
            boolean odlWithNoSubscriptions = this.isOnDemandLoadWithNoSubscriptions(service);
            if (cacheCleared && odlWithNoSubscriptions) {
                this.stopService(service.getSelfLink(), false, null);
                continue;
            }
            if (!shouldPause) continue;
            if (!cacheCleared) {
                this.clearCachedServiceState(service.getSelfLink(), null);
                if (odlWithNoSubscriptions) {
                    this.stopService(service.getSelfLink(), false, null);
                    continue;
                }
            }
            if ((existing = this.pendingPauseServices.put(service.getSelfLink(), service)) == null) {
                ++pauseServiceCount;
            }
            if ((factoryPath = UriUtils.getParentPath(service.getSelfLink())) != null) {
                this.serviceFactoriesUnderMemoryPressure.add(factoryPath);
            }
            if (deadlineMicros >= Utils.getNowMicrosUtc()) continue;
            break;
        }
        if (pauseServiceCount == 0) {
            return;
        }
        ServiceHost.ServiceHostState serviceHostState = hostState;
        synchronized (serviceHostState) {
            hostState.serviceCount = this.attachedServices.size();
        }
        this.pauseServices();
    }

    private void pauseServices() {
        if (this.host.isStopping()) {
            return;
        }
        ServiceHost.ServiceHostState hostState = this.host.getStateNoCloning();
        int servicePauseCount = 0;
        for (Service s : this.pendingPauseServices.values()) {
            if (s.getProcessingStage() != Service.ProcessingStage.AVAILABLE) continue;
            try {
                s.setProcessingStage(Service.ProcessingStage.PAUSED);
            }
            catch (Throwable e2) {
                this.host.log(Level.INFO, "Failure setting stage to %s for %s: %s", new Object[]{Service.ProcessingStage.PAUSED, s.getSelfLink(), e2.getMessage()});
                continue;
            }
            ++servicePauseCount;
            String path = s.getSelfLink();
            this.host.sendRequest(ServiceContextIndexService.createPost(this.host, path, s).setReferer(this.host.getUri()).setCompletion((o, e) -> {
                if (e != null && !this.host.isStopping()) {
                    this.host.log(Level.WARNING, "Failure indexing service for pause: %s", Utils.toString(e));
                    this.resumeService(path, s);
                    return;
                }
                Service serviceEntry = this.pendingPauseServices.remove(path);
                if (serviceEntry == null && !this.host.isStopping()) {
                    this.host.log(Level.INFO, "aborting pause for %s", path);
                    this.resumeService(path, s);
                    this.host.processPendingServiceAvailableOperations(s, null, false);
                    return;
                }
                ServiceHost.ServiceHostState serviceHostState = hostState;
                synchronized (serviceHostState) {
                    if (null != this.attachedServices.remove(path)) {
                        --hostState.serviceCount;
                    }
                }
                this.persistedServiceLastAccessTimes.remove(path);
                this.host.getManagementService().adjustStat("servicePauseCount", 1.0);
            }));
        }
        this.host.log(Level.INFO, "Paused %d services, attached: %d, cached: %d, persistedServiceLastAccessTimes: %d", servicePauseCount, hostState.serviceCount, this.cachedServiceStates.size(), this.persistedServiceLastAccessTimes.size());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void resumeService(String path, Service resumedService) {
        ServiceHost.ServiceHostState hostState;
        if (this.host.isStopping()) {
            return;
        }
        resumedService.setHost(this.host);
        resumedService.setProcessingStage(Service.ProcessingStage.AVAILABLE);
        ServiceHost.ServiceHostState serviceHostState = hostState = this.host.getStateNoCloning();
        synchronized (serviceHostState) {
            if (!this.attachedServices.containsKey(path)) {
                this.attachedServices.put(path, resumedService);
                ++hostState.serviceCount;
            }
        }
        if (ServiceHost.isServiceIndexed(resumedService)) {
            this.persistedServiceLastAccessTimes.put(resumedService.getSelfLink(), Utils.getNowMicrosUtc());
        }
        this.host.getManagementService().adjustStat("serviceResumeCount", 1.0);
    }

    boolean checkAndResumePausedServiceOnOwner(Operation inboundOp) {
        String key = inboundOp.getUri().getPath();
        if (ServiceHost.isHelperServicePath(key)) {
            key = UriUtils.getParentPath(key);
        }
        String factoryPath = UriUtils.getParentPath(key);
        FactoryService factoryService = null;
        if (factoryPath != null) {
            factoryService = (FactoryService)this.host.findService(factoryPath);
        }
        if (factoryService != null && !this.serviceFactoriesUnderMemoryPressure.contains(factoryPath)) {
            if (factoryService.hasOption(Service.ServiceOption.ON_DEMAND_LOAD)) {
                inboundOp.addPragmaDirective("xn-check-index");
            } else {
                return false;
            }
        }
        String path = key;
        if (factoryService == null) {
            this.host.failRequestServiceNotFound(inboundOp);
            return true;
        }
        if (this.host.isStopping() && inboundOp.hasPragmaDirective("xn-no-index-update") && inboundOp.getAction() == Service.Action.DELETE) {
            inboundOp.complete();
            return true;
        }
        if (inboundOp.hasPragmaDirective("xn-check-index")) {
            Service service = this.pendingPauseServices.remove(path);
            if (service != null) {
                this.host.log(Level.INFO, "Aborting service pause for %s", path);
                this.resumeService(path, service);
                return false;
            }
            if (inboundOp.getExpirationMicrosUtc() < Utils.getNowMicrosUtc()) {
                this.host.log(Level.WARNING, "Request to %s has expired", path);
                return false;
            }
            if (this.host.isStopping()) {
                return false;
            }
            service = this.attachedServices.get(path);
            if (service != null && service.getProcessingStage() == Service.ProcessingStage.PAUSED) {
                this.host.log(Level.INFO, "Service attached, but paused, aborting pause for %s", path);
                this.resumeService(path, service);
                return false;
            }
            long pendingPauseCount = this.pendingPauseServices.size();
            if (pendingPauseCount == 0L) {
                return this.host.checkAndOnDemandStartService(inboundOp, factoryService);
            }
            this.host.schedule(() -> {
                this.host.log(Level.INFO, "Retrying index lookup for %s, pending pause: %d", path, pendingPauseCount);
                this.checkAndResumePausedServiceOnOwner(inboundOp);
            }, 1L, TimeUnit.SECONDS);
            return true;
        }
        inboundOp.addPragmaDirective("xn-check-index");
        OperationContext inputContext = OperationContext.getOperationContext();
        Operation query = ServiceContextIndexService.createGet(this.host, path).setCompletion((o, e) -> {
            OperationContext.setFrom(inputContext);
            if (e != null) {
                this.host.log(Level.WARNING, "Failure checking if service paused: " + Utils.toString(e), new Object[0]);
                this.host.handleRequest(null, inboundOp);
                return;
            }
            if (!o.hasBody()) {
                this.host.handleRequest(null, inboundOp);
                return;
            }
            Service resumedService = (Service)o.getBodyRaw();
            this.resumeService(path, resumedService);
            this.host.handleRequest(null, inboundOp);
        });
        OperationContext.setAuthorizationContext(this.host.getSystemAuthorizationContext());
        this.host.sendRequest(query.setReferer(this.host.getUri()));
        return true;
    }

    public void close() {
        this.pendingPauseServices.clear();
        this.cachedServiceStates.clear();
        this.persistedServiceLastAccessTimes.clear();
    }

    private boolean isTransactional(Operation op) {
        return op != null && op.getTransactionId() != null && this.host.getTransactionServiceUri() != null;
    }

    private boolean isOnDemandLoadWithNoSubscriptions(Service service) {
        if (!service.hasOption(Service.ServiceOption.ON_DEMAND_LOAD)) {
            return false;
        }
        UtilityService utilityService = (UtilityService)service.getUtilityService("/subscriptions");
        return !utilityService.hasSubscribers();
    }

    static class CachedServiceStateKey {
        private String servicePath;
        private String transactionId;

        CachedServiceStateKey(String servicePath, String transactionId) {
            this.servicePath = servicePath;
            this.transactionId = transactionId;
        }

        public int hashCode() {
            return Objects.hash(this.servicePath, this.transactionId);
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof CachedServiceStateKey) {
                CachedServiceStateKey that = (CachedServiceStateKey)o;
                return Objects.equals(this.servicePath, that.servicePath) && Objects.equals(this.transactionId, that.transactionId);
            }
            return false;
        }

        public String toString() {
            return String.format("CachedServiceStateKey{servicePath: %s, transactionId: %s}", this.servicePath, this.transactionId);
        }
    }
}

