// Copyright 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.content_shell;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.drawable.ClipDrawable;
import android.support.annotation.NonNull;
import android.util.AttributeSet;
import android.view.ActionMode;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.view.inputmethod.EditorInfo;
import android.view.inputmethod.InputMethodManager;
import android.widget.EditText;
import android.widget.FrameLayout;
import android.widget.ImageButton;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.TextView.OnEditorActionListener;

import com.vccorp.base.annotations.CalledByNative;
import com.vccorp.content.R;
import com.vccorp.content.browser.JavascriptInterface;
import com.vccorp.content_public.browser.JavascriptInjector;
import com.vccorp.content_public.browser.SelectionPopupController;
import com.vccorp.content_shell.ShellViewAndroidDelegate;
import com.vccorp.object.LinkVisited;
import com.vccorp.object.UserInfo;
import com.vccorp.ui.base.ActivityWindowAndroid;

import org.chromium.base.Callback;
import org.chromium.base.annotations.JNINamespace;
import org.chromium.components.embedder_support.view.ContentView;
import org.chromium.components.embedder_support.view.ContentViewRenderView;
import org.chromium.content_public.browser.ActionModeCallbackHelper;
import org.chromium.content_public.browser.LoadUrlParams;
import org.chromium.content_public.browser.NavigationController;
import org.chromium.content_public.browser.WebContents;
import org.chromium.ui.base.ViewAndroidDelegate;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import static org.chromium.content_shell.ShellManager.mStartupUrl;

/**
 * Container for the various UI components that make up a shell window.
 */
@JNINamespace("content")
public class Shell extends LinearLayout {

    private static final long COMPLETED_PROGRESS_TIMEOUT_MS = 200;
    private static final String NAME_ANONYMOUS = "anonymous";
    private static final int ID_ANONYMOUS = 123;

    private final Runnable mClearProgressRunnable = new Runnable() {
        @Override
        public void run() {
            mProgressDrawable.setLevel(0);
        }
    };
    private static String CLIENT_ID = "2407e7c8032ed1fba1cfe6a305178519";
    private static String CLIENT_SECRET = "qwe9667cd5c50c42d68f36fe4df9poiu";

    private WebContents mWebContents;
    private NavigationController mNavigationController;
    private EditText mUrlTextView;
    private ImageButton mPrevButton;
    private ImageButton mNextButton;
    private ImageButton mStopReloadButton;

    private ClipDrawable mProgressDrawable;

    private long mNativeShell;
    private ContentViewRenderView mContentViewRenderView;
    private ActivityWindowAndroid mWindow;
    private ShellViewAndroidDelegate mViewAndroidDelegate;

    private boolean mLoading;
    private boolean mIsFullscreen;


    private Callback<Boolean> mOverlayModeChangedCallbackForTesting;
    private boolean isUpdated;
    private OnWebViewListener onWebViewListener;
    private long timeLoadPage;
    private String TAG = Shell.class.getName();
    private UserInfo userInfoVCCInfo;
    private OnConfigListener configListener;
    private DatabaseHandler databaseHandler;

    /**
     * Constructor for inflating via XML.
     */
    public Shell(Context context, AttributeSet attrs) {
        super(context, attrs);
        SecureManager.shareInstance().setContext(context);

    }

    /**
     * Set the SurfaceView being renderered to as soon as it is available.
     */
    public void setContentViewRenderView(ContentViewRenderView contentViewRenderView) {
        FrameLayout contentViewHolder = (FrameLayout) findViewById(R.id.contentview_holder);

        if (contentViewRenderView == null) {
            if (mContentViewRenderView != null) {
                contentViewHolder.removeView(mContentViewRenderView);
            }
        } else {
            contentViewHolder.addView(contentViewRenderView,
                    new FrameLayout.LayoutParams(
                            FrameLayout.LayoutParams.MATCH_PARENT,
                            FrameLayout.LayoutParams.MATCH_PARENT));
        }
        mContentViewRenderView = contentViewRenderView;


    }

    /**
     * Initializes the Shell for use.
     *
     * @param nativeShell The pointer to the native Shell object.
     * @param window      The owning window for this shell.
     */
    public void initialize(long nativeShell, ActivityWindowAndroid window) {
        mNativeShell = nativeShell;
        mWindow = window;
        databaseHandler = new DatabaseHandler(getContext());
    }

    /**
     * Closes the shell and cleans up the native instance, which will handle destroying all
     * dependencies.
     */
    public void close() {
        if (mNativeShell == 0) return;
        nativeCloseShell(mNativeShell);
    }

    @CalledByNative
    private void onNativeDestroyed() {
        mWindow = null;
        mNativeShell = 0;
        mWebContents = null;
    }

    /**
     * @return Whether the Shell has been destroyed.
     * @see #onNativeDestroyed()
     */
    public boolean isDestroyed() {
        return mNativeShell == 0;
    }

    /**
     * @return Whether or not the Shell is loading content.
     */
    public boolean isLoading() {
        return mLoading;
    }

    @Override
    protected void onFinishInflate() {
        super.onFinishInflate();

        View toolbar = findViewById(R.id.toolbar);
        mProgressDrawable = (ClipDrawable) toolbar.getBackground();
        initializeUrlField();
        initializeNavigationButtons();
    }

    private void initializeUrlField() {
        mUrlTextView = (EditText) findViewById(R.id.url);
        mUrlTextView.setOnEditorActionListener(new OnEditorActionListener() {
            @Override
            public boolean onEditorAction(TextView v, int actionId, KeyEvent event) {
                if ((actionId != EditorInfo.IME_ACTION_GO) && (event == null
                        || event.getKeyCode() != KeyEvent.KEYCODE_ENTER
                        || event.getAction() != KeyEvent.ACTION_DOWN)) {
                    return false;
                }
                if (mWebContents != null) {
                    loadUrl(mUrlTextView.getText().toString());
                }
                setKeyboardVisibilityForUrl(false);
                getContentView().requestFocus();
                return true;
            }
        });
        mUrlTextView.setOnFocusChangeListener(new OnFocusChangeListener() {
            @Override
            public void onFocusChange(View v, boolean hasFocus) {
                setKeyboardVisibilityForUrl(hasFocus);
                mNextButton.setVisibility(hasFocus ? GONE : VISIBLE);
                mPrevButton.setVisibility(hasFocus ? GONE : VISIBLE);
                mStopReloadButton.setVisibility(hasFocus ? GONE : VISIBLE);
                if (!hasFocus) {
                    mUrlTextView.setText(mWebContents.getVisibleUrl());
                }
            }
        });
        mUrlTextView.setOnKeyListener(new OnKeyListener() {
            @Override
            public boolean onKey(View v, int keyCode, KeyEvent event) {
                if (keyCode == KeyEvent.KEYCODE_BACK) {
                    getContentView().requestFocus();
                    return true;
                }
                return false;
            }
        });
    }

    /**
     * Loads an URL.  This will perform minimal amounts of sanitizing of the URL to attempt to
     * make it valid.
     *
     * @param url The URL to be loaded by the shell.
     */
    public void loadUrl(String url) {
        SecureManager.shareInstance().checkStateSecure(getContext(), configListener);

        if (url == null) return;

//        if (url.equals(mWebContents.getLastCommittedUrl())) {
//            mNavigationController.reload(true);
//        } else {
        if (mNavigationController != null) {
            mNavigationController.loadUrl(new LoadUrlParams(sanitizeUrl(url)));
        }
//        }


        mUrlTextView.clearFocus();
        // TODO(aurimas): Remove this when crbug.com/174541 is fixed.
        getContentView().clearFocus();
        getContentView().requestFocus();
        if (onWebViewListener != null) {
            onWebViewListener.onLoading(url);
        }
    }

    /**
     * Given an URL, this performs minimal sanitizing to ensure it will be valid.
     *
     * @param url The url to be sanitized.
     * @return The sanitized URL.
     */
    public static String sanitizeUrl(String url) {
        if (url == null) return null;
        if (url.startsWith("www.") || url.indexOf(":") == -1) url = "http://" + url;
        return url;
    }

    private void initializeNavigationButtons() {
        mPrevButton = (ImageButton) findViewById(R.id.prev);
        mPrevButton.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mNavigationController.canGoBack()) mNavigationController.goBack();
            }
        });

        mNextButton = (ImageButton) findViewById(R.id.next);
        mNextButton.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mNavigationController.canGoForward()) mNavigationController.goForward();
            }
        });
        mStopReloadButton = (ImageButton) findViewById(R.id.stop_reload_button);
        mStopReloadButton.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mLoading) mWebContents.stop();
                else mNavigationController.reload(true);
            }
        });
    }

    @SuppressWarnings("unused")
    @CalledByNative
    private void onUpdateUrl(String url) {
        mUrlTextView.setText(url);

        LinkVisited linkVisited = null;
        if (databaseHandler != null) {
            if (userInfoVCCInfo == null) {
                userInfoVCCInfo = new UserInfo(ID_ANONYMOUS, NAME_ANONYMOUS,
                        NAME_ANONYMOUS, NAME_ANONYMOUS,
                        String.valueOf(System.currentTimeMillis()));
                databaseHandler.addUser(userInfoVCCInfo);
            }
            if (getWebContents().getLastCommittedUrl() != null) {
                linkVisited = new LinkVisited(getWebContents().getVisibleUrl(),
                        getWebContents().getLastCommittedUrl(), getWebContents().getTitle(),
                        userInfoVCCInfo.getUid(),
                        userInfoVCCInfo.getTimeCreate());
            }
            databaseHandler.addLinkVisited(linkVisited);

        }

        if (onWebViewListener != null) {
            onWebViewListener.updateURL(url);
        }
        mStartupUrl = url;
    }

    @SuppressWarnings("unused")
    @CalledByNative
    private void onLoadProgressChanged(double progress) {
        removeCallbacks(mClearProgressRunnable);

        mProgressDrawable.setLevel((int) (10000.0 * progress));
        if (progress == 1.0) {
            postDelayed(mClearProgressRunnable, COMPLETED_PROGRESS_TIMEOUT_MS);
            if (onWebViewListener != null) {
                onWebViewListener.onLoaded();
            }
        }

    }

    private String convertMinisecondToDate(String currentTimeMillis) {
        @SuppressLint("SimpleDateFormat") SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(Long.parseLong(currentTimeMillis));
        return simpleDateFormat.format(calendar.getTime());

    }

    @CalledByNative
    private void toggleFullscreenModeForTab(boolean enterFullscreen) {
        mIsFullscreen = enterFullscreen;
        LinearLayout toolBar = (LinearLayout) findViewById(R.id.toolbar);
        toolBar.setVisibility(enterFullscreen ? GONE : VISIBLE);
    }

    @CalledByNative
    private boolean isFullscreenForTabOrPending() {
        return mIsFullscreen;
    }

    @SuppressWarnings("unused")
    @CalledByNative
    private void setIsLoading(boolean loading) {
        mLoading = loading;
        if (mLoading) {
            if (onWebViewListener != null) {
                onWebViewListener.onStartLoading();
            }
            timeLoadPage = System.currentTimeMillis();
            mStopReloadButton
                    .setImageResource(android.R.drawable.ic_menu_close_clear_cancel);
        } else {
            mStopReloadButton.setImageResource(R.drawable.ic_refresh);
            if (onWebViewListener != null) {
                onWebViewListener.timeLoadPage(System.currentTimeMillis() - timeLoadPage);
            }

        }
    }

    public ShellViewAndroidDelegate getViewAndroidDelegate() {
        return mViewAndroidDelegate;
    }

    /**
     * Initializes the ContentView based on the native org.chromium.tab contents pointer passed in.
     *
     * @param webContents A {@link WebContents} object.
     */
    @SuppressWarnings("unused")
    @CalledByNative
    private void initFromNativeTabContents(WebContents webContents) {
        Context context = getContext();
        ContentView cv = ContentView.createContentView(context, webContents);
//        cv.setCurrentUrl(mStartupUrl);
        mViewAndroidDelegate = new ShellViewAndroidDelegate(cv);
        webContents.initialize(
                "", mViewAndroidDelegate, cv, mWindow, WebContents.createDefaultInternalsHolder());
        mWebContents = webContents;


        JavascriptInjector ji = JavascriptInjector.fromWebContents(mWebContents);
        ji.addPossiblyUnsafeInterface(new MyJavaScriptInterface(context), "ViVaHandler", null);


        SelectionPopupController.fromWebContents(webContents)
                .setActionModeCallback(defaultActionCallback());
        mNavigationController = mWebContents.getNavigationController();
        if (getParent() != null)
            mWebContents.onShow();

        if (mWebContents.getVisibleUrl() != null) {
            mUrlTextView.setText(mWebContents.getVisibleUrl());
        }
        ((FrameLayout) findViewById(R.id.contentview_holder)).addView(cv,
                new FrameLayout.LayoutParams(
                        FrameLayout.LayoutParams.MATCH_PARENT,
                        FrameLayout.LayoutParams.MATCH_PARENT));
        cv.requestFocus();
        mContentViewRenderView.setCurrentWebContents(mWebContents);
    }

    /**
     * {link @ActionMode.Callback} that uses the default implementation in
     * {@link SelectionPopupController}.
     */
    private ActionMode.Callback defaultActionCallback() {
        final ActionModeCallbackHelper helper =
                SelectionPopupController.fromWebContents(mWebContents)
                        .getActionModeCallbackHelper();

        return new ActionMode.Callback() {
            @Override
            public boolean onCreateActionMode(ActionMode mode, Menu menu) {
                helper.onCreateActionMode(mode, menu);
                return true;
            }

            @Override
            public boolean onPrepareActionMode(ActionMode mode, Menu menu) {
                return helper.onPrepareActionMode(mode, menu);
            }

            @Override
            public boolean onActionItemClicked(ActionMode mode, MenuItem item) {
                return helper.onActionItemClicked(mode, item);
            }

            @Override
            public void onDestroyActionMode(ActionMode mode) {
                helper.onDestroyActionMode();
            }

        };
    }

    @CalledByNative
    public void setOverlayMode(boolean useOverlayMode) {
        mContentViewRenderView.setOverlayVideoMode(useOverlayMode);
        if (mOverlayModeChangedCallbackForTesting != null) {
            mOverlayModeChangedCallbackForTesting.onResult(useOverlayMode);
        }
    }

    @CalledByNative
    public void sizeTo(int width, int height) {
        mWebContents.setSize(width, height);
    }

    public void setOverayModeChangedCallbackForTesting(Callback<Boolean> callback) {
        mOverlayModeChangedCallbackForTesting = callback;
    }

    /**
     * Enable/Disable navigation(Prev/Next) button if navigation is allowed/disallowed
     * in respective direction.
     *
     * @param controlId Id of button to update
     * @param enabled   enable/disable value
     */
    @CalledByNative
    private void enableUiControl(int controlId, boolean enabled) {
        if (controlId == 0) {
            mPrevButton.setEnabled(enabled);
        } else if (controlId == 1) {
            mNextButton.setEnabled(enabled);
        }
    }

    /**
     * @return The {@link ViewGroup} currently shown by this Shell.
     */
    public ViewGroup getContentView() {
        if (mWebContents.getViewAndroidDelegate() != null) {
            ViewAndroidDelegate viewDelegate = mWebContents.getViewAndroidDelegate();
            return viewDelegate != null ? viewDelegate.getContainerView() : null;
        }
        return null;
    }

    /**
     * @return The {@link WebContents} currently managing the content shown by this Shell.
     */
    public WebContents getWebContents() {
        return mWebContents;
    }

    private void setKeyboardVisibilityForUrl(boolean visible) {
        InputMethodManager imm = (InputMethodManager) getContext().getSystemService(
                Context.INPUT_METHOD_SERVICE);
        if (visible) {
            imm.showSoftInput(mUrlTextView, InputMethodManager.SHOW_IMPLICIT);
        } else {
            imm.hideSoftInputFromWindow(mUrlTextView.getWindowToken(), 0);
        }
    }

    public void onListenVivaUserInfo(UserInfo userInfo) {
        String jsString = getContext().getResources().getString(R.string.user_infor_test);

        JSONObject person = new JSONObject();
        try {
            if (userInfo != null) {
                person.put("code", userInfo == null ? 404 : 200);

                person.put("message", "Đăng nhập thành công");

                JSONObject data = new JSONObject();
                data.put("id", userInfo.getUid());
                data.put("Name", userInfo.getName());
                data.put("email", userInfo.getEmail());
                person.put("data", data);
            }

        } catch (JSONException e) {
            e.printStackTrace();
        }
        final String webUrl = "javascript:onListenVivaUserInfo(" + person + ")";
        // Add this to avoid android.view.windowmanager$badtokenexception unable to add window
        post(new Runnable() {
            @Override
            public void run() {
                loadUrl(webUrl);
            }
        });
    }

    public void setConfigListener(OnConfigListener configListener) {
        this.configListener = configListener;
    }

//    @Override
//    public void onSuccess(AccessToken accessToken) {
//        this.accessToken = accessToken;
//        Log.d(TAG, "AccessToken: " + accessToken.getAccessToken());
//        getUserInforFromSDK();
//
//    }
//
//    @Override
//    public void onFailed(KingIdException e) {
//        Log.d(TAG, "KingIdException: " + e.getErrorCode());
//
//    }
//
//    @Override
//    public void onCancel() {
//        Log.d(TAG, "onCancel: ");
//
//    }

    public class MyJavaScriptInterface {

        private Context context;

        public MyJavaScriptInterface(Context context) {
            this.context = context;
        }

        @JavascriptInterface
        public void getVivaUserInfo() {
            getUserInforFromSDK();

        }
    }

    private void getUserInforFromSDK() {
        onListenVivaUserInfo(userInfoVCCInfo);
    }


    public String getCurrentUrl() {
        return mUrlTextView.getText().toString();
    }

    public boolean setUserInfor(@NonNull UserInfo userInfoInfor) {
        if (SecureManager.shareInstance().isAllowUsingBrowser()) {
            userInfoVCCInfo = userInfoInfor;
            return databaseHandler.addUser(userInfoInfor);
        }

        return false;

    }


    public void setOnWebViewListener(OnWebViewListener onWebViewListener) {
        this.onWebViewListener = onWebViewListener;
    }

    public boolean removeUserById(int uId) {
        if (databaseHandler != null) {
            return databaseHandler.removeUserByUserId(uId);
        } else return false;
    }

    public boolean removeUserByName(String name) {
        if (databaseHandler != null) {
            return databaseHandler.removeUserByName(name);
        } else return false;
    }

    public boolean removeUserByEmail(String email) {
        if (databaseHandler != null) {
            return databaseHandler.removeUserByEmail(email);
        } else return false;
    }


    public void setRemoveAllHistoryAfter30Days(boolean toggle) {
        if (toggle) {
            if (databaseHandler != null) {
                databaseHandler.removeHistory30Days();
            }
        }
    }

    public void removeHistoryById(int userId) {
        if (databaseHandler != null) {
            databaseHandler.removeHistoryByUserId(userId);
        }
    }

    public void removeHistoryByRange(Date start, Date end) {
        if (databaseHandler != null) {
            databaseHandler.removeHistoryByRange(start, end);
        }
    }

    public List<LinkVisited> getAllHistoryByUser(int userId) {
        if (databaseHandler != null) {
            return databaseHandler.getAllHistoryOfUser(userId);
        }
        return new ArrayList<>();
    }

    public List<LinkVisited> getAllHistory() {
        if (databaseHandler != null) {
            databaseHandler.getAllHistory();
        }
        return new ArrayList<>();
    }

    public List<LinkVisited> getAllHistoryByRange(Date start, Date end) {
        if (databaseHandler != null) {
            return databaseHandler.getHistoryByRange(start, end);
        }
        return new ArrayList<>();
    }


    private static native void nativeCloseShell(long shellPtr);
}
