// Copyright 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

package org.chromium.content_shell;

import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.Log;
import android.view.LayoutInflater;
import android.widget.FrameLayout;
import android.widget.Toast;

import com.vccorp.base.annotations.CalledByNative;
import com.vccorp.base.library_loader.LibraryProcessType;
import com.vccorp.base.library_loader.ProcessInitException;
import com.vccorp.content.R;
import com.vccorp.content_public.browser.BrowserStartupController;
import com.vccorp.content_public.browser.DeviceUtils;

import com.vccorp.ui.base.ActivityWindowAndroid;

import org.chromium.base.CommandLine;
import org.chromium.base.ThreadUtils;
import org.chromium.base.annotations.JNINamespace;
import org.chromium.base.library_loader.LibraryLoader;
import org.chromium.components.embedder_support.view.ContentViewRenderView;
import org.chromium.content_public.browser.WebContents;
import org.chromium.ui.base.WindowAndroid;

import static com.vccorp.SHWebViewController.COMMAND_LINE_ARGS_KEY;
import static com.vccorp.SHWebViewController.COMMAND_LINE_FILE;

/**
 * Container and generator of ShellViews.
 */
@JNINamespace("content")
public class ShellManager extends FrameLayout {

    private static final String ACTIVE_SHELL_URL_KEY = "ACTIVE_SHELL_URL_KEY";
    private static final String TAG = ShellManager.class.getName();
    public static String DEFAULT_SHELL_URL = "https://www.google.com.vn/";
    private static final String STRING_BLANK = "about://blank";
    private ActivityWindowAndroid mWindow;
    private Shell mActiveShell;
    private static final String RUN_WEB_TESTS_SWITCH = "run-web-tests";
    public static String mStartupUrl = STRING_BLANK;
    private ActivityWindowAndroid mWindowAndroid;
    // The target for all content rendering.
    private ContentViewRenderView mContentViewRenderView;
    private boolean isInited = false;
    private OnWebViewListener onWebViewListener;
    private OnConfigListener configListener;

    public ShellManager(Context context) {
        super(context);
        initView();
    }

    /**
     * Constructor for inflating via XML.
     */


    public ShellManager(final Context context, AttributeSet attrs) {
        super(context, attrs);
        initView();
    }

    private void initView() {
        if (!CommandLine.isInitialized()) {
            initCommandLine();
            String[] commandLineParams = getCommandLineParamsFromIntent(new Intent());
            if (commandLineParams != null) {
                CommandLine.getInstance().appendSwitchesAndArguments(commandLineParams);
            }
        }

        DeviceUtils.addDeviceSpecificUserAgentSwitch();

        try {
            LibraryLoader.getInstance().ensureInitialized(LibraryProcessType.PROCESS_WEBVIEW);
        } catch (ProcessInitException e) {
            Log.e(TAG, "ContentView initialization failed.", e);
            // Since the library failed to initialize nothing in the application
            // can work, so kill the whole application not just the activity
            System.exit(-1);
            return;
        }
        nativeInit(this);
        settingView(new Bundle());
    }

    public void finishInitialization(Bundle savedInstanceState) {
        String shellUrl;
        if (!TextUtils.isEmpty(mStartupUrl)) {
            shellUrl = mStartupUrl;
        } else {
            shellUrl = ShellManager.DEFAULT_SHELL_URL;
        }

        if (savedInstanceState != null
                && savedInstanceState.containsKey(ACTIVE_SHELL_URL_KEY)) {
            shellUrl = savedInstanceState.getString(ACTIVE_SHELL_URL_KEY);
        }
        launchShell(shellUrl);
    }

    public void initializationFailed() {
        Log.e(TAG, "ContentView initialization failed.");
        Toast.makeText(getContext(),
                R.string.browser_process_initialization_failed,
                Toast.LENGTH_SHORT).show();
//        finish();
    }


    /**
     * @param window The window used to generate all shells.
     */
    public void setWindow(ActivityWindowAndroid window) {
        assert window != null;
        mWindow = window;
        mContentViewRenderView = new ContentViewRenderView(getContext());
        mContentViewRenderView.onNativeLibraryLoaded(window);
    }

    /**
     * @return The window used to generate all shells.
     */
    public WindowAndroid getWindow() {
        return mWindow;
    }

    /**
     * Get the ContentViewRenderView.
     */
    public ContentViewRenderView getContentViewRenderView() {
        return mContentViewRenderView;
    }

    /**
     * Sets the startup URL for new shell windows.
     */
    public void setStartupUrl(String url) {
        mStartupUrl = url;
    }

    /**
     * @return The currently visible shell view or null if one is not showing.
     */
    public Shell getActiveShell() {
        return mActiveShell;
    }

    /**
     * Creates a new shell pointing to the specified URL.
     *
     * @param url The URL the shell should load upon creation.
     */
    public void launchShell(String url) {
        if (!TextUtils.isEmpty(url)) {
            ThreadUtils.assertOnUiThread();
//            if (mActiveShell == null) {
//                mStartupUrl = url;
//                return;
//            }
            Shell previousShell = mActiveShell;
            nativeLaunchShell(url);
            if (previousShell != null)
                previousShell.close();
        }

    }

    @SuppressWarnings("unused")
    @CalledByNative
    private Object createShell(long nativeShellPtr) {
        //
        if (mContentViewRenderView == null) {
            mContentViewRenderView = new ContentViewRenderView(getContext());
            mContentViewRenderView.onNativeLibraryLoaded(mWindow);
        }
        LayoutInflater inflater =
                (LayoutInflater) getContext().getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        Shell shellView = (Shell) inflater.inflate(R.layout.shell_view, null);
        shellView.initialize(nativeShellPtr, mWindow);
        shellView.setOnWebViewListener(onWebViewListener);
        shellView.setConfigListener(configListener);
        onWebViewListener.onShellReady(shellView);
        // TODO(tedchoc): Allow switching back to these inactive shells.
        if (mActiveShell != null) {
            removeShell(mActiveShell);
        }

        showShell(shellView);
        return shellView;
    }

    public void initCommandLine() {
        if (!CommandLine.isInitialized()) {
            CommandLine.initFromFile(COMMAND_LINE_FILE);
        }
    }


    private void showShell(Shell shellView) {
        shellView.setContentViewRenderView(mContentViewRenderView);
        addView(shellView, new FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT));
        mActiveShell = shellView;
        WebContents webContents = mActiveShell.getWebContents();

        if (webContents != null) {
            mContentViewRenderView.setCurrentWebContents(webContents);
            webContents.onShow();
        }
    }

    @CalledByNative
    private void removeShell(Shell shellView) {
        if (shellView == mActiveShell) mActiveShell = null;
        if (shellView.getParent() == null) return;
        shellView.setContentViewRenderView(null);
        removeView(shellView);
        onWebViewListener.onRemoveShell();
    }

    /**
     * Destroys the Shell manager and associated components.
     * Always called at activity exit, and potentially called by native in cases where we need to
     * control the timing of mContentViewRenderView destruction. Must handle being called twice.
     */
    @CalledByNative
    public void destroy() {
        // Remove active shell (Currently single shell support only available).
        if (mActiveShell != null) {
            removeShell(mActiveShell);
        }
        if (mContentViewRenderView != null) {
            mContentViewRenderView.destroy();
            mContentViewRenderView = null;
        }
    }

    private static native void nativeInit(Object shellManagerInstance);

    private static native void nativeLaunchShell(String url);


    public static String[] getCommandLineParamsFromIntent(Intent intent) {
        return intent != null ? intent.getStringArrayExtra(COMMAND_LINE_ARGS_KEY) : null;
    }

    public void restoreBundle(Bundle savedInstanceState) {
        mWindowAndroid.restoreInstanceState(savedInstanceState);
        if (TextUtils.isEmpty(mStartupUrl)) {
            mStartupUrl = STRING_BLANK;
        }

        if (savedInstanceState != null
                && savedInstanceState.containsKey(ACTIVE_SHELL_URL_KEY)) {
            mStartupUrl = savedInstanceState.getString(ACTIVE_SHELL_URL_KEY);
            if (isInited)
                if (!TextUtils.isEmpty(mStartupUrl)
                        && !mStartupUrl.equals(STRING_BLANK)) {
                    launchShell(mStartupUrl);
                }
        }
    }

    public void saveInstanceState(Bundle outState) {
        if (getActiveShell() != null) {
            WebContents webContents = getActiveShell().getWebContents();
            if (webContents != null) {
                outState.putString(ACTIVE_SHELL_URL_KEY, webContents.getLastCommittedUrl());
            }
            mWindowAndroid.saveInstanceState(outState);
        }


    }

    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        mWindowAndroid.onActivityResult(requestCode, resultCode, data);
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        destroy();
    }


    public void settingView(Bundle savedInstanceState) {
        final boolean listenToActivityState = false;
        mWindowAndroid = new ActivityWindowAndroid(getContext(), listenToActivityState);
        mWindowAndroid.restoreInstanceState(savedInstanceState);
        setWindow(mWindowAndroid);
        // Set up the animation placeholder to be the SurfaceView. This disables the
        // SurfaceView's 'hole' clipping during animations that are notified to the window.
        mWindowAndroid.setAnimationPlaceholderView(
                getContentViewRenderView().getSurfaceView());

//        mStartupUrl = getUrlFromIntent(new Intent());
        mStartupUrl = DEFAULT_SHELL_URL;
        if (!TextUtils.isEmpty(mStartupUrl)) {
            setStartupUrl(Shell.sanitizeUrl(mStartupUrl));
        }

        if (CommandLine.getInstance().hasSwitch(RUN_WEB_TESTS_SWITCH)) {
            try {
                BrowserStartupController.get(LibraryProcessType.PROCESS_WEBVIEW)
                        .startBrowserProcessesSync(false);
            } catch (ProcessInitException e) {
                Log.e(TAG, "Failed to load native library.", e);
                System.exit(-1);
            }
        } else {
            try {
                BrowserStartupController.get(LibraryProcessType.PROCESS_WEBVIEW)
                        .startBrowserProcessesAsync(
                                true, false, new BrowserStartupController.StartupCallback() {
                                    @Override
                                    public void onSuccess() {
                                        isInited = true;
                                        finishInitialization(savedInstanceState);
                                    }

                                    @Override
                                    public void onFailure() {
                                        initializationFailed();
                                    }
                                });
            } catch (ProcessInitException e) {
                Log.e(TAG, "Unable to load native library.", e);
                System.exit(-1);
            }
        }
    }


    public void setOnWebViewListener(OnWebViewListener onWebViewListener) {
        this.onWebViewListener = onWebViewListener;
    }

    public void setConfigListener(OnConfigListener configListener) {
        this.configListener = configListener;
    }
}
