/**
 * Copyright 2014 Google Inc. All rights reserved.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.vcc.playerwrappersdk.controllers;

import android.content.Context;
import android.graphics.Color;
import android.graphics.drawable.Drawable;
import android.os.Handler;
import android.os.Looper;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;

import com.google.android.exoplayer.ExoPlayer;
import com.google.android.libraries.mediaframework.exoplayerextensions.ExoplayerWrapper;
import com.google.android.libraries.mediaframework.exoplayerextensions.Video;
import com.google.android.libraries.mediaframework.layeredvideo.PlaybackControlLayer;
import com.google.android.libraries.mediaframework.layeredvideo.SimpleVideoPlayer;
import com.google.android.libraries.mediaframework.layeredvideo.Util;
import com.vcc.playerwrappersdk.BuildConfig;
import com.vcc.playerwrappersdk.Log;
import com.vcc.playerwrappersdk.utils.EBusData;

import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.Timer;
import java.util.TimerTask;

import static com.vcc.playerwrappersdk.controllers.PlayerController.BUFFER_PROGRESS;
import static com.vcc.playerwrappersdk.controllers.PlayerController.CURRENT_TIME;
import static com.vcc.playerwrappersdk.controllers.PlayerController.DURATION_VIDEO;
import static com.vcc.playerwrappersdk.controllers.PlayerController.TIME_PROGRESS;
import static com.vcc.playerwrappersdk.utils.InvokedMethodUtils.EVENT;
import static com.vcc.playerwrappersdk.utils.InvokedMethodUtils.VALUE;

/**
 * The ImaPlayer is responsible for displaying both videos and ads. This is accomplished using two
 * video players. The content player displays the user's video. When an ad is requested, the ad
 * video player is overlaid on the content video player. When the ad is complete, the ad video
 * player is destroyed and the content video player is displayed again.
 */
public class ImaPlayer {

    private static final String PLAYER_TYPE = "google/gmf-android";
    private static final String PLAYER_VERSION = BuildConfig.VERSION_NAME;
    private static final long UPDATE_TIME = 500; //0.5s
    private boolean autoplay;

    private WeakReference<PlayerController.OnPlayerCallback> weakReferencePlayerCallback;

    /**
     * Contains the content player and the ad frame layout.
     */
    private FrameLayout container;
    /**
     * Plays the content (i.e. the actual video).
     */
    private SimpleVideoPlayer contentPlayer;
    /**
     * Listener for the content player
     */
    private final ExoplayerWrapper.PlaybackListener contentPlaybackListener
            = new ExoplayerWrapper.PlaybackListener() {

        /**
         * We don't respond to errors.
         * @param e The error.
         */
        @Override
        public void onError(Exception e) {
            Log.e("State change video", "Error video content!");
            PlayerController.OnPlayerCallback onPlayerCallback = getOnPlayerCallback();
            if (onPlayerCallback != null) {
                onPlayerCallback.onError(e.getMessage());
            }
        }

        /**
         * We notify the adLoader when the content has ended so it knows to play postroll ads.
         * @param playWhenReady Whether the video should play as soon as it is loaded.
         * @param playbackState The state of the Exoplayer instance.
         */
        @Override
        public void onStateChanged(boolean playWhenReady, int playbackState) {
            if (isAutoplay()) {
                if (playbackState == ExoPlayer.STATE_PREPARING) {
                    play();

                    cancelCountDown();

                    startCountTime();
                }
            }
//            Log.e("State change video", "State: " + playbackState);
            PlayerController.OnPlayerCallback onPlayerCallback = getOnPlayerCallback();
            if (onPlayerCallback != null)
                onPlayerCallback.playState(playbackState);

            if (playbackState == ExoPlayer.STATE_READY) {

                if (contentPlayer.isPlaying()) {
                    if (onPlayerCallback != null) {
                        onPlayerCallback.onPlayPressed();
                    } else
                        Log.e("State change video", "OnPlayerCallback is null.");
                    cancelCountDown();

                    startCountTime();
                } else {
                    if (onPlayerCallback != null)
                        onPlayerCallback.onPausePressed();
                    else
                        Log.e("State change video", "OnPlayerCallback is null.");
                    cancelCountDown();
                }
                if (onPlayerCallback != null)
                    onPlayerCallback.playState(playbackState);
                else
                    Log.e("State change video", "OnPlayerCallback is null.");
            }

            if (playbackState == ExoPlayer.STATE_ENDED) {
                cancelCountDown();
            }
            if (playbackState == ExoPlayer.STATE_BUFFERING
                    || playbackState == ExoPlayer.STATE_IDLE
                    || playbackState == ExoPlayer.STATE_PREPARING) {
                cancelCountDown();
            }
        }

        /**
         * We don't respond to size changes.
         * @param width The new width of the player.
         * @param height The new height of the player.
         * @param unappliedRotationDegrees The new rotation angle of the player thats not applied.
         */
        @Override
        public void onVideoSizeChanged(int width, int height, int unappliedRotationDegrees, float pixelWidthHeightRatio) {
            Log.e("State change video", "Video size changed!");
            PlayerController.OnPlayerCallback onPlayerCallback = getOnPlayerCallback();
            if (onPlayerCallback != null) {
                onPlayerCallback.videSizeChange(width, height, unappliedRotationDegrees, pixelWidthHeightRatio);
            }
        }
    };

    /**
     * The callback that is triggered when fullscreen mode is entered or closed.
     */
    private PlaybackControlLayer.FullscreenCallback fullscreenCallback;
    /**
     * This is the layout of the container before fullscreen mode has been entered.
     * When we leave fullscreen mode, we restore the layout of the container to this layout.
     */
    private ViewGroup.LayoutParams originalContainerLayoutParams;

    /**
     * A flag to indicate whether the ads has been shown.
     */

    private boolean adsShown;
    private Timer timer;
    private TimerTask timerTask;

    /**
     * @param mContext           The mContext that will contain the video player.
     * @param container          The {@link FrameLayout} which will contain the video player.
     * @param video              The video that should be played.
     * @param videoTitle         The title of the video (displayed on the left of the top chrome).
     * @param isFixedContent     Fixed size of content crop center.
     * @param fullscreenCallback The callback that should be triggered when the player enters or
     *                           leaves fullscreen.
     */
    public ImaPlayer(Context mContext,
                     FrameLayout container,
                     Video video,
                     String videoTitle,
                     boolean isFixedContent,
                     PlaybackControlLayer.FullscreenCallback fullscreenCallback) {

        cancelCountDown();

        this.container = container;

        setFixedContent(isFixedContent);

        contentPlayer = new SimpleVideoPlayer(mContext,
                container,
                video,
                videoTitle,
                autoplay,
                isFixedContent);

        contentPlayer.addPlaybackListener(contentPlaybackListener);

        // Move the content player's surface layer to the background so that the ad player's surface
        // layer can be overlaid on top of it during ad playback.
        contentPlayer.moveSurfaceToBackground();
        contentPlayer.hide();

        this.originalContainerLayoutParams = container.getLayoutParams();

        setFullscreenCallback(fullscreenCallback);
    }

    /**
     * @param mContext   The mContext that will contain the video player.
     * @param container  The {@link FrameLayout} which will contain the video player.
     * @param video      The video that should be played.
     * @param videoTitle The title of the video (displayed on the left of the top chrome).
     */
    public ImaPlayer(Context mContext,
                     FrameLayout container,
                     Video video,
                     String videoTitle, boolean isFixedContent) {
        this(mContext, container, video, videoTitle, isFixedContent, null);
    }

    public PlayerController.OnPlayerCallback getOnPlayerCallback() {
        return weakReferencePlayerCallback == null ? null : weakReferencePlayerCallback.get();
    }

    public void setOnPlayerCallback(PlayerController.OnPlayerCallback onPlaybackState) {
        weakReferencePlayerCallback = new WeakReference<>(onPlaybackState);
        if (contentPlayer != null) {
            contentPlayer.setOnPlayerCallback(onPlaybackState);
        }
    }

    /**
     * Pause video playback.
     */
    public void pause() {
        try {
            contentPlayer.pause();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Resume video playback.
     */
    public void play() {
        resumeContent();
    }

    /**
     * Set the logo with appears in the left of the top chrome.
     *
     * @param logo The drawable which will be the logo.
     */
    public void setLogoImage(Drawable logo) {
        contentPlayer.setLogoImage(logo);
    }

    /**
     * Sets the color of the top chrome, bottom chrome, and background.
     *
     * @param color a color derived from the @{link Color} class
     *              (ex. {@link Color#RED}).
     */
    public void setChromeColor(int color) {
        contentPlayer.setChromeColor(color);
    }

    /**
     * Sets the color of the buttons and seek bar.
     *
     * @param color a color derived from the @{link Color} class
     *              (ex. {@link Color#RED}).
     */
    public void setPlaybackControlColor(int color) {
        contentPlayer.setPlaybackControlColor(color);
    }

    /**
     * Creates a button to put in the top right of the video player.
     *
     * @param icon               The image of the action (ex. trash can).
     * @param contentDescription The text description this action. This is used in case the
     *                           action buttons do not fit in the video player. If so, an overflow
     *                           button will appear and, when clicked, it will display a list of the
     *                           content descriptions for each action.
     * @param onClickListener    The handler for when the action is triggered.
     */
    public void addActionButton(Drawable icon,
                                String contentDescription,
                                View.OnClickListener onClickListener) {
        contentPlayer.addActionButton(icon, contentDescription, onClickListener);
    }

    /**
     * Set the callback which will be called when the player enters and leaves fullscreen mode.
     *
     * @param fullscreenCallback The callback should hide other views in the mContext when the player
     *                           enters fullscreen mode and show other views when the player leaves
     *                           fullscreen mode.
     */
    public void setFullscreenCallback(
            final PlaybackControlLayer.FullscreenCallback fullscreenCallback) {
        this.fullscreenCallback = new PlaybackControlLayer.FullscreenCallback() {
            @Override
            public void onGoToFullscreen() {
                fullscreenCallback.onGoToFullscreen();
                container.setLayoutParams(Util.getLayoutParamsBasedOnParent(
                        container,
                        ViewGroup.LayoutParams.MATCH_PARENT,
                        ViewGroup.LayoutParams.MATCH_PARENT
                ));
            }

            @Override
            public void onReturnFromFullscreen() {
                fullscreenCallback.onReturnFromFullscreen();
                container.setLayoutParams(originalContainerLayoutParams);
            }
        };
        contentPlayer.setFullscreenCallback(fullscreenCallback);
    }

    /**
     * When you are finished using this {@link ImaPlayer}, make sure to call this method.
     */
    public void release() {

        contentPlayer.release();
        contentPlayer = null;
        cancelCountDown();
    }

    /**
     * Pause and hide the content player.
     */
    private void hideContentPlayer() {
        contentPlayer.pause();
        contentPlayer.hide();
    }

    /**
     * Show the content player and start playing again.
     */
    private void showContentPlayer() {
        if (contentPlayer == null)
            return;
        contentPlayer.show();
        contentPlayer.play();
    }

    private void startCountTime() {
        if (timer == null)
            timer = new Timer();
        if (timerTask == null)
            timerTask = new TimerTask() {
                private Handler handler = new Handler(Looper.getMainLooper());

                @Override
                public void run() {
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            updateProgress();
                        }
                    });
                }
            };
        timer.schedule(timerTask, 0, UPDATE_TIME);
    }

    private void cancelCountDown() {
        if (timerTask != null) {
            timerTask.cancel();
        }
        if (timer != null) {
            timer.cancel();
            timer.purge();
        }
        timer = null;
        timerTask = null;
    }

    public void handleMessage() {

    }

    protected void updateProgress() {
        if (contentPlayer == null)
            return;

        int position = contentPlayer.getCurrentPosition();
        int duration = contentPlayer.getDuration();
        int buffering = contentPlayer.getCurrentBuffering();

        if (duration > 0) {
            pushMessage(TIME_PROGRESS, "" + position * 100 / duration);
            pushMessage(BUFFER_PROGRESS, "" + buffering * 100 / duration);
            pushMessage(CURRENT_TIME, miliSecondsToTimer(position));
            pushMessage(DURATION_VIDEO, miliSecondsToTimer(duration));
        }
        PlayerController.OnPlayerCallback onPlayerCallback = getOnPlayerCallback();
        if (onPlayerCallback != null) {

            onPlayerCallback.onUpdateProgress(position, duration);

            onPlayerCallback.onBuffering(buffering);
        }
    }


    public String miliSecondsToTimer(long milliseconds) {
        String finalTimerString;
        String minutesString;
        String secondsString;

        // Convert total duration into time
        int hours = (int) (milliseconds / (1000 * 60 * 60));
        int minutes = (int) (milliseconds % (1000 * 60 * 60)) / (1000 * 60);
        int seconds = (int) ((milliseconds % (1000 * 60 * 60)) % (1000 * 60) / 1000);
        // Add hours if there

        // Prepending 0 to hours if it is one digit
        if (hours > 0 && hours < 10) {
            finalTimerString = "0" + hours + ":";
        } else if (hours >= 10) {
            finalTimerString = "" + hours + ":";
        } else {
            finalTimerString = "";
        }

        // Prepending 0 to minutes if it is one digit
        if (minutes < 10) {
            minutesString = "0" + minutes + ":";
        } else {
            minutesString = "" + minutes + ":";
        }

        // Prepending 0 to seconds if it is one digit
        if (seconds < 10) {
            secondsString = "0" + seconds;
        } else {
            secondsString = "" + seconds;
        }
        if (hours == 0 && minutes == 0 && seconds == 0)
            return "00:00";
        finalTimerString = finalTimerString + minutesString + secondsString;

        // return timer string
        return finalTimerString;
    }


    /**
     * Pause the content player and notify the ad callbacks that the content has paused.
     */
    private void pauseContent() {
        hideContentPlayer();
    }

    /**
     * Resume the content and notify the ad callbacks that the content has resumed.
     */
    private void resumeContent() {
        showContentPlayer();
    }


    public void setPlayCallBack(PlaybackControlLayer.PlayCallback playCallBack) {
        contentPlayer.setPlayCallback(playCallBack);
    }

    public boolean isAutoplay() {
        return autoplay;
    }

    public void setAutoplay(boolean autoplay) {
        this.autoplay = autoplay;
    }

    public void hideController(boolean shouldHide) {
        contentPlayer.hideController(shouldHide);
    }

    public void setHideTitle(boolean shouldHide) {
        if (contentPlayer != null) {
            contentPlayer.setHideTitle(shouldHide);
            if (shouldHide)
                contentPlayer.hideTopChrome();
        }
    }

    public int getCurrentPosition() {
        return contentPlayer.getCurrentPosition();
    }

    public int getLengthVideo() {
        return contentPlayer.getDuration();
    }

    public void toggleFullscreen() {
        if (contentPlayer != null)
            contentPlayer.toggleFullscreen();
    }

    public void seekTo(int position) {
        if (position > getLengthVideo())
            position = getLengthVideo();
        contentPlayer.seekTo(position);
        contentPlayer.play();
    }

    public boolean isPlaying() {
        return contentPlayer.isPlaying();
    }

    public void setFixedContent(boolean fixedContent) {
    }

    public boolean isMute() {
        return (contentPlayer != null) && (contentPlayer.getCurrentVolume() == 0);
    }

    public boolean setMute(boolean mute) {
        return contentPlayer != null && contentPlayer.setMute(mute);
    }


    private void pushMessage(String event, Object value) {
        try {
            JSONObject messageInit = new JSONObject();
            messageInit.put(EVENT, event);
            messageInit.put(VALUE, value);
            EBusData.getInstance().postEvent(messageInit);
        } catch (JSONException e) {
        }
    }


}
