/**
 * Copyright 2014 Google Inc. All rights reserved.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.android.libraries.mediaframework.layeredvideo;

import android.content.Context;
import android.graphics.Matrix;
import android.util.AttributeSet;
import android.view.TextureView;
import android.widget.FrameLayout;

/**
 * A SurfaceView implementation that resizes itself according to a specified aspect ratio.
 */
public class VideoTextureView extends TextureView {

    /**
     * The surface view will not resize itself if the fractional difference
     * between its default aspect ratio and the aspect ratio of the video falls
     * below this threshold.
     */
    private static final float MAX_ASPECT_RATIO_DEFORMATION_PERCENT = 0.01f;

    /**
     * The ratio of the width and height of the video.
     */
    private float videoAspectRatio;
    private FrameLayout layoutContainer;
    private boolean fixedContent;
    private float mVideoWidth;
    private float mVideoHeight;

    /**
     * @param context The context (ex {@link android.app.Activity}) that created this object.
     */
    public VideoTextureView(Context context) {
        super(context);
    }

    /**
     * @param context The context (ex {@link android.app.Activity}) that created this object.
     * @param attrs   A set of attributes to configure the {@link android.view.SurfaceView}.
     */
    public VideoTextureView(Context context, AttributeSet attrs) {
        super(context, attrs);
    }

    /**
     * Resize the view based on the width and height specifications.
     *
     * @param widthMeasureSpec  The specified width.
     * @param heightMeasureSpec The specified height.
     */
    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
        if (isFixedContent()) {
            if (mVideoWidth == 0 && mVideoHeight == 0)
                return;
            if (videoAspectRatio == 0)
                return;
            float viewWidth = layoutContainer.getWidth();
            float viewHeight = layoutContainer.getHeight();
            updateTextureViewSize(viewWidth, viewHeight);
            return;
        }

        int width = getMeasuredWidth();
        int height = getMeasuredHeight();
        if (videoAspectRatio != 0) {
            float viewAspectRatio = (float) width / height;
            float aspectDeformation = videoAspectRatio / viewAspectRatio - 1;
            if (aspectDeformation > MAX_ASPECT_RATIO_DEFORMATION_PERCENT) {
                height = (int) (width / videoAspectRatio);
            } else if (aspectDeformation < -MAX_ASPECT_RATIO_DEFORMATION_PERCENT) {
                width = (int) (height * videoAspectRatio);
            }
        }
        setMeasuredDimension(width, height);
    }

    private void updateTextureViewSize(float viewWidth, float viewHeight) {
        float scaleX = 1.0f;
        float scaleY = 1.0f;

        if (mVideoWidth > viewWidth && mVideoHeight > viewHeight) {
            scaleX = mVideoWidth / viewWidth;
            scaleY = mVideoHeight / viewHeight;
        } else if (mVideoWidth < viewWidth && mVideoHeight < viewHeight) {
            scaleY = viewWidth / mVideoWidth;
            scaleX = viewHeight / mVideoHeight;
        } else if (viewWidth > mVideoWidth) {
            scaleY = (viewWidth / mVideoWidth) / (viewHeight / mVideoHeight);
        } else if (viewHeight > mVideoHeight) {
            scaleX = (viewHeight / mVideoHeight) / (viewWidth / mVideoWidth);
        }

        // Calculate pivot points, in our case crop from center
        float pivotPointX = viewWidth / 2.f;
        float pivotPointY = viewHeight / 2.f;

        Matrix matrix = new Matrix();
        matrix.setScale(scaleX, scaleY, pivotPointX, pivotPointY);

        setTransform(matrix);
    }

    /**
     * Set the aspect ratio that this {@link VideoTextureView} should satisfy.
     *
     * @param widthHeightRatio The width to height ratio.
     */
    public void setVideoWidthHeightRatio(float widthHeightRatio) {
        if (this.videoAspectRatio != widthHeightRatio) {
            this.videoAspectRatio = widthHeightRatio;
            requestLayout();
        }
    }

    public void setLayoutContainer(FrameLayout layoutContainer) {
        this.layoutContainer = layoutContainer;
    }

    public void setFixedContent(boolean fixedContent) {
        this.fixedContent = fixedContent;
    }

    public boolean isFixedContent() {
        return fixedContent;
    }

    public void setVideoSize(float width, float height) {
        if (mVideoWidth == 0 && mVideoHeight == 0) {
            mVideoWidth = width;
            mVideoHeight = height;
        }
    }
}
