/*
 * Copyright (C) 2000-2025 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */

package com.vaadin.data;

import java.io.Serializable;
import java.util.Objects;
import java.util.function.BiFunction;

import com.vaadin.server.SerializablePredicate;
import com.vaadin.shared.ui.ErrorLevel;

/**
 * A functional interface for validating user input or other potentially invalid
 * data. When a validator instance is applied to a value of the corresponding
 * type, it returns a <i>result</i> signifying that the value either passed or
 * failed the validation.
 * <p>
 * For instance, the following validator checks if a number is positive:
 *
 * <pre>
 * Validator&lt;Integer&gt; v = num -&gt; {
 *     if (num &gt;= 0)
 *         return ValidationResult.ok();
 *     else
 *         return ValidationResult.error("number must be positive");
 * };
 * </pre>
 *
 * @author Vaadin Ltd.
 *
 * @since 8.0
 *
 * @param <T>
 *            the type of the value to validate
 *
 * @see ValidationResult
 */
@FunctionalInterface
public interface Validator<T>
        extends BiFunction<T, ValueContext, ValidationResult>, Serializable {

    /**
     * Validates the given value. Returns a {@code ValidationResult} instance
     * representing the outcome of the validation.
     *
     * @param value
     *            the input value to validate
     * @param context
     *            the value context for validation
     * @return the validation result
     */
    @Override
    public ValidationResult apply(T value, ValueContext context);

    /**
     * Returns a validator that passes any value.
     *
     * @param <T>
     *            the value type
     * @return an always-passing validator
     */
    public static <T> Validator<T> alwaysPass() {
        return (value, context) -> ValidationResult.ok();
    }

    /**
     * Builds a validator out of a conditional function and an error message. If
     * the function returns true, the validator returns {@code Result.ok()}; if
     * it returns false or throws an exception,
     * {@link ValidationResult#error(String)} is returned with the given message
     * and error level {@link ErrorLevel#ERROR}.
     * <p>
     * For instance, the following validator checks if a number is between 0 and
     * 10, inclusive:
     *
     * <pre>
     * Validator&lt;Integer&gt; v = Validator.from(num -&gt; num &gt;= 0 &amp;&amp; num &lt;= 10,
     *         "number must be between 0 and 10");
     * </pre>
     *
     * @param <T>
     *            the value type
     * @param guard
     *            the function used to validate, not null
     * @param errorMessage
     *            the message returned if validation fails, not null
     * @return the new validator using the function
     */
    public static <T> Validator<T> from(SerializablePredicate<T> guard,
            String errorMessage) {
        Objects.requireNonNull(errorMessage, "errorMessage cannot be null");
        return from(guard, ctx -> errorMessage);
    }

    /**
     * Builds a validator out of a conditional function and an error message. If
     * the function returns true, the validator returns {@code Result.ok()}; if
     * it returns false or throws an exception,
     * {@link ValidationResult#error(String)} is returned with the given message
     * and error level.
     * <p>
     * For instance, the following validator checks if a number is between 0 and
     * 10, inclusive:
     *
     * <pre>
     * Validator&lt;Integer&gt; v = Validator.from(num -&gt; num &gt;= 0 &amp;&amp; num &lt;= 10,
     *         "number must be between 0 and 10", ErrorLevel.ERROR);
     * </pre>
     *
     * @param <T>
     *            the value type
     * @param guard
     *            the function used to validate, not null
     * @param errorMessage
     *            the message returned if validation fails, not null
     * @param errorLevel
     *            the error level for failures from this validator, not null
     * @return the new validator using the function
     *
     * @since 8.2
     */
    public static <T> Validator<T> from(SerializablePredicate<T> guard,
            String errorMessage, ErrorLevel errorLevel) {
        Objects.requireNonNull(errorMessage, "errorMessage cannot be null");
        return from(guard, ctx -> errorMessage, errorLevel);
    }

    /**
     * Builds a validator out of a conditional function and an error message
     * provider. If the function returns true, the validator returns
     * {@code Result.ok()}; if it returns false or throws an exception,
     * {@code Result.error()} is returned with the message from the provider.
     *
     * @param <T>
     *            the value type
     * @param guard
     *            the function used to validate, not null
     * @param errorMessageProvider
     *            the provider to generate error messages, not null
     * @return the new validator using the function
     */
    public static <T> Validator<T> from(SerializablePredicate<T> guard,
            ErrorMessageProvider errorMessageProvider) {
        return from(guard, errorMessageProvider, ErrorLevel.ERROR);
    }

    /**
     * Builds a validator out of a conditional function and an error message
     * provider. If the function returns true, the validator returns
     * {@code Result.ok()}; if it returns false or throws an exception,
     * {@code Result.error()} is returned with the message from the provider.
     *
     * @param <T>
     *            the value type
     * @param guard
     *            the function used to validate, not null
     * @param errorMessageProvider
     *            the provider to generate error messages, not null
     * @param errorLevel
     *            the error level for failures from this validator, not null
     * @return the new validator using the function
     *
     * @since 8.2
     */
    public static <T> Validator<T> from(SerializablePredicate<T> guard,
            ErrorMessageProvider errorMessageProvider, ErrorLevel errorLevel) {
        Objects.requireNonNull(guard, "guard cannot be null");
        Objects.requireNonNull(errorMessageProvider,
                "errorMessageProvider cannot be null");
        Objects.requireNonNull(errorLevel, "errorLevel cannot be null");
        return (value, context) -> {
            try {
                if (guard.test(value)) {
                    return ValidationResult.ok();
                }
                return ValidationResult.create(
                        errorMessageProvider.apply(context), errorLevel);
            } catch (Exception e) {
                return ValidationResult.create(
                        errorMessageProvider.apply(context), errorLevel);
            }
        };
    }
}
