/*
 * Copyright (C) 2000-2025 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.ui;

import java.io.Serializable;

/**
 * Internal utility class.
 *
 * @since 8.1
 * @author Vaadin Ltd
 */
public class ComponentRootSetter implements Serializable {

    private ComponentRootSetter() {
        // Util methods only
    }

    /**
     * Sets the composition root for the given custom component or composite.
     * <p>
     * For internal use only.
     *
     * @param customComponent
     *            the custom component or composite
     * @param component
     *            the component to assign as composition root
     */
    public static void setRoot(Component customComponent, Component component) {
        if (customComponent instanceof CustomComponent) {
            ((CustomComponent) customComponent).setCompositionRoot(component);
        } else if (customComponent instanceof Composite) {
            ((Composite) customComponent).setCompositionRoot(component);
        } else {
            throw new IllegalArgumentException(
                    "Parameter is of an unsupported type: "
                            + customComponent.getClass().getName());
        }
    }

    /**
     * Checks if the given custom component or composite may accept a root
     * component.
     * <p>
     * For internal use only.
     *
     * @param customComponent
     *            the custom component or composite
     * @return
     * @since 8.4
     *
     */
    public static boolean canSetRoot(Component customComponent) {
        if (customComponent instanceof CustomComponent) {
            return true;
        }
        if (customComponent instanceof Composite) {
            return ((Composite) customComponent).getCompositionRoot() == null;
        }
        return false;
    }
}
