/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.ui.renderers;

import com.vaadin.shared.ui.grid.renderers.TextRendererState;

import elemental.json.Json;
import elemental.json.JsonValue;

/**
 * A renderer for presenting a plain text representation of any value.
 * {@link Object#toString()} is used for determining the text to show.
 *
 * @since 7.4
 * @author Vaadin Ltd
 */
public class TextRenderer extends AbstractRenderer<Object, Object> {

    /**
     * Creates a new text renderer that uses <code>""</code> to represent null
     * values.
     */
    public TextRenderer() {
        this("");
    }

    /**
     * Creates a new text renderer with the given string to represent null
     * values.
     *
     * @param nullRepresentation
     *            the textual representation of {@code null} value
     */
    public TextRenderer(String nullRepresentation) {
        super(Object.class, nullRepresentation);
    }

    @Override
    public JsonValue encode(Object value) {
        if (value == null) {
            return super.encode(null);
        }
        String stringValue = value.toString();
        if (stringValue == null) {
            return super.encode(null);
        }
        return Json.create(stringValue);
    }

    @Override
    public String getNullRepresentation() {
        return super.getNullRepresentation();
    }

    @Override
    protected TextRendererState getState() {
        return (TextRendererState) super.getState();
    }

    @Override
    protected TextRendererState getState(boolean markAsDirty) {
        return (TextRendererState) super.getState(markAsDirty);
    }
}
