/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.ui.declarative.converters;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Locale;

import com.vaadin.data.Converter;
import com.vaadin.data.Result;
import com.vaadin.data.ValueContext;
import com.vaadin.ui.declarative.DesignAttributeHandler;

/**
 * A {@link LocalDate} converter to be used by {@link DesignAttributeHandler}.
 * Provides ISO-compliant way of storing date and time.
 *
 * @since 8.0
 * @author Vaadin Ltd
 */
public class DesignLocalDateConverter implements Converter<String, LocalDate> {

    @Override
    public Result<LocalDate> convertToModel(String value,
            ValueContext context) {
        for (String pattern : new String[] { "yyyy-MM-dd", "yyyy-MM",
                "yyyy" }) {
            try {
                Locale effectiveLocale = context.getLocale()
                        .orElse(Locale.ENGLISH);
                LocalDate date = DateTimeFormatter
                        .ofPattern(pattern, effectiveLocale)
                        .parse(value, LocalDate::from);
                return Result.ok(date);
            } catch (DateTimeParseException e) {
                // not parseable, ignore and try another format
            }
        }
        return Result.error("Could not parse date value: " + value);
    }

    @Override
    public String convertToPresentation(LocalDate value, ValueContext context) {
        return DateTimeFormatter
                .ofPattern("yyyy-MM-dd",
                        context.getLocale().orElse(Locale.ENGLISH))
                .format(value);
    }

}
