/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.ui.components.grid;

import java.io.Serializable;

import com.vaadin.data.Binder;
import com.vaadin.shared.Registration;
import com.vaadin.ui.Grid;

/**
 * An editor in a Grid.
 *
 * @author Vaadin Ltd
 * @since 8.0
 *
 * @param <T>
 */
public interface Editor<T> extends Serializable {

    /**
     * Sets the underlying Binder to this Editor.
     *
     * @param binder
     *            the binder for updating editor fields; not {@code null}
     * @return this editor
     */
    public Editor<T> setBinder(Binder<T> binder);

    /**
     * Returns the underlying Binder from Editor.
     *
     * @return the binder; not {@code null}
     */
    public Binder<T> getBinder();

    /**
     * Sets the Editor buffered mode. When the editor is in buffered mode, edits
     * are only committed when the user clicks the save button. In unbuffered
     * mode valid changes are automatically committed.
     *
     * @param buffered
     *            {@code true} if editor should be buffered; {@code false} if
     *            not
     * @return this editor
     */
    public Editor<T> setBuffered(boolean buffered);

    /**
     * Enables or disabled the Editor. A disabled editor cannot be opened.
     *
     * @param enabled
     *            {@code true} if editor should be enabled; {@code false} if not
     * @return this editor
     */
    public Editor<T> setEnabled(boolean enabled);

    /**
     * Returns whether Editor is buffered or not.
     *
     * @see #setBuffered(boolean)
     *
     * @return {@code true} if editor is buffered; {@code false} if not
     */
    public boolean isBuffered();

    /**
     * Returns whether Editor is enabled or not.
     *
     * @return {@code true} if editor is enabled; {@code false} if not
     */
    public boolean isEnabled();

    /**
     * Returns whether Editor is open or not.
     *
     * @return {@code true} if editor is open; {@code false} if not
     */
    public boolean isOpen();

    /**
     * Saves any changes from the Editor fields to the edited bean.
     *
     * @return {@code true} if save succeeded; {@code false} if not
     */
    public boolean save();

    /**
     * Close the editor discarding any unsaved changes.
     */
    public void cancel();

    /**
     * Opens the editor interface for the provided row. Scrolls the Grid to
     * bring the row to view if it is not already visible.
     *
     * Note that any cell content rendered by a WidgetRenderer will not be
     * visible in the editor row.
     *
     * @see #setEnabled(boolean)
     * @since 8.2
     *
     * @param rowNumber
     *            the row number of the edited item
     * @throws IllegalStateException
     *             if the editor is not enabled, Grid is read-only, or already
     *             editing a different item in buffered mode
     * @throws IllegalArgumentException
     *             if the {@code rowNumber} is not in the backing data provider
     */
    public void editRow(int rowNumber);

    /**
     * Sets the caption of the save button in buffered mode.
     *
     * @param saveCaption
     *            the save button caption
     * @return this editor
     */
    public Editor<T> setSaveCaption(String saveCaption);

    /**
     * Sets the caption of the cancel button in buffered mode.
     *
     * @param cancelCaption
     *            the cancel button caption
     * @return this editor
     */
    public Editor<T> setCancelCaption(String cancelCaption);

    /**
     * Gets the caption of the save button in buffered mode.
     *
     * @return the save button caption
     */
    public String getSaveCaption();

    /**
     * Gets the caption of the cancel button in buffered mode.
     *
     * @return the cancel button caption
     */
    public String getCancelCaption();

    /**
     * Sets the error message generator for this editor.
     * <p>
     * The default message is a concatenation of column field validation
     * failures and bean validation failures.
     *
     * @param errorGenerator
     *            the function to generate error messages; not {@code null}
     * @return this editor
     *
     * @see EditorErrorGenerator
     */
    public Editor<T> setErrorGenerator(EditorErrorGenerator<T> errorGenerator);

    /**
     * Gets the error message generator of this editor.
     *
     * @return the function that generates error messages; not {@code null}
     *
     * @see EditorErrorGenerator
     */
    public EditorErrorGenerator<T> getErrorGenerator();

    /**
     * Adds an editor save {@code listener}.
     *
     * @param listener
     *            save listener
     * @return a registration object for removing the listener
     */
    public Registration addSaveListener(EditorSaveListener<T> listener);

    /**
     * Adds an editor cancel {@code listener}.
     *
     * @param listener
     *            cancel listener
     * @return a registration object for removing the listener
     */
    public Registration addCancelListener(EditorCancelListener<T> listener);

    /**
     * Adds an editor open {@code listener}.
     *
     * @param listener
     *            open listener
     * @return a registration object for removing the listener
     *
     * @since 8.1
     */
    public Registration addOpenListener(EditorOpenListener<T> listener);

    /**
     * Gets the Grid instance which this editor belongs to.
     *
     * @return the grid which owns the editor
     */
    public Grid<T> getGrid();
}
