/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.ui;

import java.time.LocalDate;

import com.vaadin.shared.ui.datefield.InlineDateFieldState;

/**
 * A date entry component, which displays the actual date selector inline.
 *
 * @see AbstractLocalDateField
 * @see DateField
 * @author Vaadin Ltd.
 * @since 8.0
 */
public class InlineDateField extends AbstractLocalDateField {

    /**
     * Constructs an empty <code>InlineDateField</code> with no caption.
     */
    public InlineDateField() {
        super();
    }

    /**
     * Constructs a new <code>InlineDateField</code> with the given caption and
     * initial text contents.
     *
     * @param caption
     *            the caption <code>String</code> for the editor.
     * @param value
     *            the LocalDate value.
     */
    public InlineDateField(String caption, LocalDate value) {
        super(caption, value);
    }

    /**
     * Constructs an empty <code>InlineDateField</code> with caption.
     *
     * @param caption
     *            the caption of the datefield.
     */
    public InlineDateField(String caption) {
        super(caption);
    }

    /**
     * Constructs a new {@code InlineDateField} with a value change listener.
     * <p>
     * The listener is called when the value of this {@code InlineDateField} is
     * changed either by the user or programmatically.
     *
     * @param valueChangeListener
     *            the value change listener, not {@code null}
     */
    public InlineDateField(ValueChangeListener<LocalDate> valueChangeListener) {
        super();
        addValueChangeListener(valueChangeListener);
    }

    /**
     * Constructs a new {@code InlineDateField} with the given caption and a
     * value change listener.
     * <p>
     * The listener is called when the value of this {@code InlineDateField} is
     * changed either by the user or programmatically.
     *
     * @param caption
     *            the caption for the field
     * @param valueChangeListener
     *            the value change listener, not {@code null}
     */
    public InlineDateField(String caption,
            ValueChangeListener<LocalDate> valueChangeListener) {
        this(valueChangeListener);
        setCaption(caption);
    }

    /**
     * Constructs a new {@code InlineDateField} with the given caption, initial
     * text contents and a value change listener.
     * <p>
     * The listener is called when the value of this {@code InlineDateField} is
     * changed either by the user or programmatically.
     *
     * @param caption
     *            the caption for the field
     * @param value
     *            the value for the field, not {@code null}
     * @param valueChangeListener
     *            the value change listener, not {@code null}
     */
    public InlineDateField(String caption, LocalDate value,
            ValueChangeListener<LocalDate> valueChangeListener) {
        this(caption, value);
        addValueChangeListener(valueChangeListener);
    }

    @Override
    protected InlineDateFieldState getState() {
        return (InlineDateFieldState) super.getState();
    }

    @Override
    protected InlineDateFieldState getState(boolean markAsDirty) {
        return (InlineDateFieldState) super.getState(markAsDirty);
    }

}
