/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.flow.internal;

import com.vaadin.flow.internal.StringUtil;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.BiPredicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileIOUtils {
    private FileIOUtils() {
    }

    public static boolean deleteFileQuietly(File file) {
        if (file == null) {
            return false;
        }
        try {
            return file.delete();
        }
        catch (Exception ignored) {
            return false;
        }
    }

    public static String urlToString(URL url) throws IOException {
        try (InputStream input = url.openStream();){
            String string = StringUtil.toUTF8String(input);
            return string;
        }
    }

    public static void copyDirectory(File source, File target) throws IOException {
        FileIOUtils.copyDirectory(source.toPath(), target.toPath(), null);
    }

    public static void copyDirectory(File source, File target, FileFilter filter) throws IOException {
        FileIOUtils.copyDirectory(source.toPath(), target.toPath(), filter);
    }

    private static void copyDirectory(final Path source, final Path target, final FileFilter filter) throws IOException {
        Files.walkFileTree(source, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                if (filter != null && !filter.accept(dir.toFile())) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                Path targetDir = target.resolve(source.relativize(dir));
                Files.createDirectories(targetDir, new FileAttribute[0]);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (filter == null || filter.accept(file.toFile())) {
                    Files.copy(file, target.resolve(source.relativize(file)), StandardCopyOption.REPLACE_EXISTING);
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static void delete(File file) throws IOException {
        if (!file.exists()) {
            return;
        }
        Path path = file.toPath();
        if (Files.isDirectory(path, new LinkOption[0])) {
            Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.delete(file);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    Files.delete(dir);
                    return FileVisitResult.CONTINUE;
                }
            });
        } else {
            Files.delete(path);
        }
    }

    public static File getUserDirectory() {
        return new File(System.getProperty("user.home"));
    }

    public static List<File> listFiles(File directory, String[] extensions, boolean recursive) throws IOException {
        ArrayList<File> result = new ArrayList<File>();
        if (!directory.isDirectory()) {
            return result;
        }
        List<Object> extensionList = extensions != null ? Arrays.asList(extensions) : List.of();
        try (Stream<Path> stream = recursive ? Files.walk(directory.toPath(), new FileVisitOption[0]) : Files.list(directory.toPath());){
            stream.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(path -> {
                if (extensionList.isEmpty()) {
                    return true;
                }
                String fileName = path.getFileName().toString();
                int lastDot = fileName.lastIndexOf(46);
                if (lastDot == -1) {
                    return false;
                }
                String extension = fileName.substring(lastDot + 1);
                return extensionList.contains(extension);
            }).forEach(path -> result.add(path.toFile()));
        }
        return result;
    }

    public static boolean contentEquals(InputStream input1, InputStream input2) throws IOException {
        return Arrays.equals(input1.readAllBytes(), input2.readAllBytes());
    }

    public static void closeQuietly(AutoCloseable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }

    public static String removeExtension(String filename) {
        int lastSeparator;
        if (filename == null) {
            return null;
        }
        int lastDot = filename.lastIndexOf(46);
        if (lastDot > (lastSeparator = Math.max(filename.lastIndexOf(47), filename.lastIndexOf(92))) && lastDot > 0) {
            return filename.substring(0, lastDot);
        }
        return filename;
    }

    public static boolean wildcardMatch(String text, String pattern) {
        if (text == null || pattern == null) {
            return false;
        }
        String regex = pattern.replace(".", "\\.").replace("*", ".*").replace("?", ".");
        return Pattern.matches(regex, text);
    }

    public static boolean isEmptyDirectory(Path directory) throws IOException {
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            return false;
        }
        try (Stream<Path> stream = Files.list(directory);){
            boolean bl = stream.findAny().isEmpty();
            return bl;
        }
    }

    public static boolean writeIfChanged(File file, List<String> content) throws IOException {
        return FileIOUtils.writeIfChanged(file, content.stream().collect(Collectors.joining("\n")));
    }

    public static boolean writeIfChanged(File file, String content) throws IOException {
        String existingFileContent = FileIOUtils.getExistingFileContent(file);
        if (content.equals(existingFileContent)) {
            FileIOUtils.log().debug("skipping writing to file '{}' because content matches", (Object)file);
            return false;
        }
        FileIOUtils.log().debug("writing to file '{}' because content does not match", (Object)file);
        file.getParentFile().mkdirs();
        Files.writeString(file.toPath(), (CharSequence)content, new OpenOption[0]);
        return true;
    }

    private static Logger log() {
        return LoggerFactory.getLogger(FileIOUtils.class);
    }

    private static String getExistingFileContent(File file) throws IOException {
        if (!file.exists()) {
            return null;
        }
        return Files.readString(file.toPath());
    }

    public static File getProjectFolderFromClasspath() {
        try {
            URL url = FileIOUtils.class.getClassLoader().getResource(".");
            if (url != null && url.getProtocol().equals("file")) {
                return FileIOUtils.getProjectFolderFromClasspath(url);
            }
        }
        catch (Exception e) {
            FileIOUtils.log().warn("Unable to determine project folder using classpath", (Throwable)e);
        }
        return null;
    }

    static File getProjectFolderFromClasspath(URL rootFolder) throws URISyntaxException {
        Path path = Path.of(rootFolder.toURI());
        if (path.endsWith(Path.of("target", "classes"))) {
            return path.getParent().getParent().toFile();
        }
        return null;
    }

    public static boolean isProbablyTemporaryFile(File file) {
        return file.getName().endsWith("~");
    }

    public static List<Path> getFilesByPattern(Path baseDir, String pattern) throws IOException {
        if (baseDir == null || !baseDir.toFile().exists()) {
            throw new IllegalArgumentException("Base directory is empty or doesn't exist: " + String.valueOf(baseDir));
        }
        if (pattern == null || pattern.isBlank()) {
            pattern = "*";
        }
        final PathMatcher matcher = FileSystems.getDefault().getPathMatcher("glob:" + pattern);
        final ArrayList<Path> matchingPaths = new ArrayList<Path>();
        Files.walkFileTree(baseDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                if (matcher.matches(file)) {
                    matchingPaths.add(file);
                }
                return FileVisitResult.CONTINUE;
            }
        });
        return matchingPaths;
    }

    public static boolean compareIgnoringIndentationAndEOL(String content1, String content2, BiPredicate<String, String> compareFn) {
        return compareFn.test(FileIOUtils.replaceIndentationAndEOL(content1), FileIOUtils.replaceIndentationAndEOL(content2));
    }

    public static boolean compareIgnoringIndentationEOLAndWhiteSpace(String content1, String content2, BiPredicate<String, String> compareFn) {
        return compareFn.test(FileIOUtils.replaceWhiteSpace(FileIOUtils.replaceIndentationAndEOL(content1)), FileIOUtils.replaceWhiteSpace(FileIOUtils.replaceIndentationAndEOL(content2)));
    }

    private static String replaceIndentationAndEOL(String text) {
        return text.replace("\r\n", "\n").replaceFirst("\n$", "").replaceAll("(?m)^(\\s)+", "");
    }

    private static String replaceWhiteSpace(String text) {
        for (String character : Stream.of("{", "}", ":", "'", "[", "]").toList()) {
            text = FileIOUtils.replaceWhiteSpaceAround(text, character);
        }
        return text;
    }

    private static String replaceWhiteSpaceAround(String text, String character) {
        return text.replaceAll(String.format("(\\s)*\\%s", character), character).replaceAll(String.format("\\%s(\\s)*", character), character);
    }
}

