/*
 * Copyright (C) 2000-2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.flow.server.frontend;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Objects;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.flow.server.ExecutionFailedException;

import static com.vaadin.flow.server.frontend.FrontendUtils.WEBPACK_CONFIG;

/**
 * Notifies the user about the existence of webpack.config.js while the project
 * is running Vite as the frontend build tool. This can be helpful especially
 * when migrating to 23.2 and later since it prevent any confusion or any
 * accidental misconfiguration in webpack related config files while using Vite.
 * <p>
 * This task is only added to the list of executables for the application
 * startup if Vite is the active frontend build tool.
 */
public class TaskNotifyWebpackConfExistenceWhileUsingVite
        implements FallibleCommand, Serializable {
    //@formatter:off
    static final String ERROR_MESSAGE =
            "%n%n**************************************************************************"
            + "%n*  Webpack related config file 'webpack.config.js' is detected in your   *"
            + "%n*  project while Vite is the default frontend build tool as of V23.2.0.  *"
            + "%n*  This error is to prevent any confusion or any accidental              *"
            + "%n*  misconfiguration in webpack related config files while using Vite.    *"
            + "%n*  Apparently, the 'webpack.config.js' file in your project seems to     *"
            + "%n*  have custom configurations in it which may not work out of the box in *"
            + "%n*  Vite.                                                                 *"
            + "%n*  If you are sure that your custom webpack configuration can be ignored *"
            + "%n*  it is advised to create backup from it and then delete/move the       *"
            + "%n*  'webpack.config.js' to be able to start the application.              *"
            + "%n*  If you are migrating from an earlier version and you need custom      *"
            + "%n*  configurations in your 'webpack.config.js' file, you should either    *"
            + "%n*  consider migrating them to a Vite alternative solution and remove     *"
            + "%n*  the 'webpack.config.js' before running your application again,        *"
            + "%n*  or just reactivate webpack via setting the                            *"
            + "%n*  'com.vaadin.experimental.webpackForFrontendBuild=true' feature flag   *"
            + "%n*  in [project-root]/src/main/resources/vaadin-featureflags.properties   *"
            + "%n*  (you may create the file if not exists) and restart the application.  *"
            + "%n*  Using webpack with Vaadin applications is deprecated and the support  *"
            + "%n*  for reactivating it would be removed in the next major release.       *"
            + "%n**************************************************************************%n%n";

    static final String WARNING_MESSAGE =
            "%n%n**************************************************************************"
            + "%n*  Webpack related config file 'webpack.config.js' is detected in your   *"
            + "%n*  project while Vite is the default frontend build tool as of V23.2.0.  *"
            + "%n*  This warning is to prevent any confusion or any accidental            *"
            + "%n*  misconfiguration in webpack related config files while using Vite.    *"
            + "%n*  Apparently, the 'webpack.config.js' file in your project seems not to *"
            + "%n*  have any custom changes/configurations (it contains the default       *"
            + "%n*  content which is generated by Vaadin). Thus to be sure, you can first *"
            + "%n*  make a backup and then safely delete it to get rid of this warning.   *"
            + "%n*  If you are migrating from an earlier version and you are still unsure *"
            + "%n*  about using Vite, you can reactivate webpack via setting the          *"
            + "%n*  'com.vaadin.experimental.webpackForFrontendBuild=true' feature flag   *"
            + "%n*  in [project-root]/src/main/resources/vaadin-featureflags.properties   *"
            + "%n*  (you can create the file if not exists) and restart the application.  *"
            + "%n*  Using webpack with Vaadin applications is deprecated and the support  *"
            + "%n*  for reactivating it would be removed in the next major release.       *"
            + "%n**************************************************************************%n%n";
    //@formatter:on

    private final File configFolder;

    TaskNotifyWebpackConfExistenceWhileUsingVite(File configFolder) {
        this.configFolder = configFolder;
    }

    @Override
    public void execute() throws ExecutionFailedException {
        try {
            validate();
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void validate() throws IOException, ExecutionFailedException {
        Path webpackConfigFilePath = Paths.get(configFolder.getPath(),
                WEBPACK_CONFIG);
        if (!Files.exists(webpackConfigFilePath)) {
            return;
        }

        long projectWebpackConfigContentHash = calculateProjectWebpackConfigContentHashCode(
                webpackConfigFilePath);
        long templateWebpackConfigContentHash = calculateTemplateWebpackConfigContentHashCode();

        if (projectWebpackConfigContentHash != templateWebpackConfigContentHash) {
            // custom content exists in the project's webpack.config.js file:
            throw new ExecutionFailedException(String.format(ERROR_MESSAGE));
        }

        log().warn(String.format(WARNING_MESSAGE));
    }

    private long calculateTemplateWebpackConfigContentHashCode()
            throws IOException {
        String fileContent = IOUtils.toString(
                Objects.requireNonNull(
                        getClass().getResourceAsStream("/" + WEBPACK_CONFIG)),
                StandardCharsets.UTF_8);
        return computeHashCode(fileContent);
    }

    private long calculateProjectWebpackConfigContentHashCode(
            Path webpackConfigFilePath) throws IOException {
        String fileContent = Files.readString(webpackConfigFilePath);
        return computeHashCode(fileContent);
    }

    private long computeHashCode(String content) {
        content = removeComments(content);
        content = removeWhiteSpaces(content);
        content = removeEmptyModuleExports(content);
        content = removeDefaults(content);
        return content.hashCode();
    }

    private String removeDefaults(String content) {
        return content.replace("constmerge=require('webpack-merge');", "")
                .replace("const{merge}=require('webpack-merge');", "")
                .replace("constflowDefaults=require('./webpack.generated.js');",
                        "")
                .replace("module.exports=merge(flowDefaults,);", "");
    }

    private String removeEmptyModuleExports(String content) {
        return content.replace("merge(flowDefaults,{});",
                "merge(flowDefaults,);");
    }

    private String removeWhiteSpaces(String content) {
        return content.replaceAll("\\s", "");
    }

    private String removeComments(String content) {
        return content.replaceAll(
                "(?:/\\*(?:[^*]|(?:\\*+[^*/]))*\\*+/)|(?://.*)", "");
    }

    private Logger log() {
        return LoggerFactory
                .getLogger(TaskNotifyWebpackConfExistenceWhileUsingVite.class);
    }
}
