import java.util.Collection;
import java.util.Optional;
import java.util.stream.Collectors;

import com.vaadin.flow.spring.security.AuthenticationContext;

import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service
public class AppUserDetailsService implements UserDetailsService {

    public record AppUserInfo(User user) implements UserDetails {

        public static Optional<User> getCurrentUser(AuthenticationContext context) {
            return context.getAuthenticatedUser(AppUserInfo.class).map(AppUserInfo::user);
        }

        @Override
        public String getUsername() {
            return user.getUsername();
        }

        @Override
        public String getPassword() {
            return user.getHashedPassword();
        }

        @Override
        public Collection<? extends GrantedAuthority> getAuthorities() {
            return user.getRoles().stream().map(role -> new SimpleGrantedAuthority("ROLE_" + role))
                    .collect(Collectors.toList());
        }
    }

    private final UserRepository userRepository;

    public AppUserDetailsService(UserRepository userRepository) {
        this.userRepository = userRepository;
    }

    @Override
    @Transactional
    public UserDetails loadUserByUsername(String username) throws UsernameNotFoundException {
        return userRepository.findByUsername(username).map(AppUserInfo::new)
                .orElseThrow(() -> new UsernameNotFoundException("No user present with username: " + username));
    }
}
