package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.internal.ComponentTracker;

/**
 * Information about a component in the source code.
 */
public record ComponentTypeAndSourceLocation(Class<? extends Component> type, List<Class<?>> inheritanceChain,
        // component might be null, it is only required for data provider handling.
        Component component, Optional<File> javaFile, Optional<ComponentTracker.Location> createLocationInProject,
        Optional<ComponentTracker.Location> attachLocationInProject, ComponentTypeAndSourceLocation parent,
        List<ComponentTypeAndSourceLocation> children, boolean createdStatically) {

    /**
     * Creates a constructor with createdStatically
     */
    public ComponentTypeAndSourceLocation(Class<? extends Component> type, List<Class<?>> inheritanceChain,
            Component component, Optional<File> javaFile, Optional<ComponentTracker.Location> createLocationInProject,
            Optional<ComponentTracker.Location> attachLocationInProject, ComponentTypeAndSourceLocation parent,
            List<ComponentTypeAndSourceLocation> children) {
        this(type, inheritanceChain, component, javaFile, createLocationInProject, attachLocationInProject, parent,
                children, false);
    }

    /**
     * Creates the entry where component is set {@code null}
     */
    public ComponentTypeAndSourceLocation(Class<? extends Component> type, List<Class<?>> inheritanceChain,
            Optional<File> javaFile, Optional<ComponentTracker.Location> createLocationInProject,
            Optional<ComponentTracker.Location> attachLocationInProject, ComponentTypeAndSourceLocation parent,
            List<ComponentTypeAndSourceLocation> children) {
        this(type, inheritanceChain, null, javaFile, createLocationInProject, attachLocationInProject, parent,
                children);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        ComponentTypeAndSourceLocation that = (ComponentTypeAndSourceLocation) o;
        return Objects.equals(javaFile, that.javaFile) && Objects.equals(type, that.type)
                && Objects.equals(parent, that.parent)
                && Objects.equals(createLocationInProject, that.createLocationInProject)
                && Objects.equals(attachLocationInProject, that.attachLocationInProject)
                && Objects.equals(children, that.children);
    }

    @Override
    public int hashCode() {
        return Objects.hash(type, javaFile, attachLocationInProject, attachLocationInProject, parent, children);
    }

    public ComponentTracker.Location getCreateLocationOrThrow() {
        return createLocationInProject()
                .orElseThrow(() -> new IllegalArgumentException("Component has no create location in the project"));
    }

    public ComponentTracker.Location getAttachLocationOrThrow() {
        return attachLocationInProject()
                .orElseThrow(() -> new IllegalArgumentException("Component has no attach location in the project"));
    }
}