package com.vaadin.copilot;

import static com.vaadin.copilot.UserInfoHandler.PRODUCT_VAADIN_COPILOT;

import java.util.concurrent.atomic.AtomicReference;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.pro.licensechecker.LicenseChecker;
import com.vaadin.pro.licensechecker.PreTrial;
import com.vaadin.pro.licensechecker.PreTrialCreationException;
import com.vaadin.pro.licensechecker.Product;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Copilot 7-day preview handler. It has two basic functionalities that start a
 * new preview and get the preview state.
 */
public class PreviewHandler extends CopilotCommand {
    private static final String REMAINING_TIME_IN_MILLIS = "remainingTimeInMillis";
    private static final String PREVIEW_ACTIVATED = "previewActivated";
    private static final String PREVIEW_ACTIVE = "active";
    private static final AtomicReference<PreTrial> latestPreview = new AtomicReference<>();

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (!(command.equals("start-preview") || command.equals("get-preview"))) {
            return false;
        }

        if (!data.has(KEY_REQ_ID)) {
            getLogger().warn("Missing required parameter 'reqId' in message: {}", data);
            devToolsInterface.send(Copilot.PREFIX + command + "-response", JacksonUtils.createObjectNode());
            return true;
        }
        ObjectNode responseData = JacksonUtils.createObjectNode();
        responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asText());

        if (command.equals("start-preview")) {
            try {
                PreTrial preTrial = startPreview();
                putPreviewData(responseData, preTrial);
            } catch (Exception e) {
                getLogger().debug("Could not start preview", e);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData,
                        "Could not connect server for starting the trial", e);
                return true;
            }
        }

        if (command.equals("get-preview")) {
            PreTrial preTrial = checkPreview();
            putPreviewData(responseData, preTrial);
        }
        devToolsInterface.send("copilot-get-preview-response", responseData);
        return true;
    }

    private PreTrial startPreview() {
        try {
            PreTrial preTrial = LicenseChecker.startPreTrial();
            if (preTrial.getTrialState() != PreTrial.PreTrialState.RUNNING) {
                throw new IllegalArgumentException("Starting preview failed (" + preTrial.getTrialState() + ")");
            }
            return preTrial;
        } catch (PreTrialCreationException.Expired ex) {
            return new PreTrial("", PreTrial.PreTrialState.EXPIRED, 0, 0);
        } catch (PreTrialCreationException ex) {
            throw new IllegalArgumentException("Starting preview failed", ex);
        }
    }

    private PreTrial checkPreview() {
        return LicenseChecker.getPreTrial(new Product(PRODUCT_VAADIN_COPILOT, CopilotVersion.getVersion()))
                .orElse(null);
    }

    private void putPreviewData(ObjectNode responseJson, PreTrial preTrial) {
        latestPreview.set(preTrial);

        if (preTrial != null) {
            boolean isActive = preTrial.getTrialState() == PreTrial.PreTrialState.RUNNING;
            // After renewal period PreTrial is present but state is set to START_ALLOWED
            boolean isActivated = preTrial.getTrialState() != PreTrial.PreTrialState.START_ALLOWED;
            responseJson.put(PREVIEW_ACTIVE, isActive);
            responseJson.put(PREVIEW_ACTIVATED, isActivated);

            // why days and not millis
            responseJson.put(REMAINING_TIME_IN_MILLIS, preTrial.getDaysRemaining() * 24 * 60 * 60 * 1000L);
        } else {
            responseJson.put(PREVIEW_ACTIVE, false);
            responseJson.put(PREVIEW_ACTIVATED, false);
        }
    }

    /**
     * Returns the latest preview activation response.
     *
     * @return Pretrial - the latest preview activation response
     */
    public static PreTrial getLatestPreview() {
        return latestPreview.get();
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(PreviewHandler.class);
    }
}
