package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Stream;

import com.vaadin.copilot.exception.SuggestRestartException;
import com.vaadin.copilot.javarewriter.ClassSourceStaticAnalyzer;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.dom.Element;
import com.vaadin.flow.server.VaadinSession;

import com.fasterxml.jackson.databind.JsonNode;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Finds the source location of a component in the project.
 */
public class ComponentSourceFinder {

    private static final String[] prefixesToSkip = new String[] { "com.vaadin.flow.component.internal.",
            "com.vaadin.flow.di.", "com.vaadin.flow.dom.", "com.vaadin.flow.internal.", "com.vaadin.flow.spring.",
            "com.vaadin.cdi.", "java.", "jdk.", "org.springframework.beans.", "org.jboss.weld.", };

    private final VaadinSession vaadinSession;

    private enum Type {
        CREATE, ATTACH
    }

    /**
     * Creates a new component source finder for the given session.
     *
     * @param vaadinSession
     *            the session to use
     */
    public ComponentSourceFinder(VaadinSession vaadinSession) {
        this.vaadinSession = vaadinSession;
    }

    /**
     * Finds the source location of a component.
     *
     * @param component
     *            the component to find, defined by uiId and nodeId in the given
     *            JSON object
     * @return the source location of the component
     */
    public ComponentTypeAndSourceLocation findTypeAndSourceLocation(JsonNode component) {
        return findTypeAndSourceLocation(component, false);
    }

    /**
     * Finds the source location of a component.
     *
     * @param component
     *            the component to find, defined by uiId and nodeId in the given
     *            JSON object
     * @return the source location of the component
     */
    public ComponentTypeAndSourceLocation findTypeAndSourceLocation(JsonNode component, boolean includeChildren) {
        return findTypeAndSourceLocation(component.get("uiId").asInt(), component.get("nodeId").asInt(),
                includeChildren);
    }

    /**
     * Finds the source location of a component.
     *
     * @param uiId
     *            the uiId of the component
     * @param nodeId
     *            the nodeId of the component
     * @return the source location of the component
     */
    public ComponentTypeAndSourceLocation findTypeAndSourceLocation(int uiId, int nodeId) {
        return findTypeAndSourceLocation(uiId, nodeId, false);
    }

    /**
     * Finds the source location of a component.
     *
     * @param uiId
     *            the uiId of the component
     * @param nodeId
     *            the nodeId of the component
     * @param includeChildren
     *            whether to include children in the search
     * @return the source location of the component
     */
    public ComponentTypeAndSourceLocation findTypeAndSourceLocation(int uiId, int nodeId, boolean includeChildren) {
        AtomicReference<ComponentTypeAndSourceLocation> info = new AtomicReference<>(null);

        vaadinSession.accessSynchronously(() -> {
            Element element = vaadinSession.findElement(uiId, nodeId);
            Optional<Component> c = element.getComponent();
            if (c.isPresent()) {
                ComponentTypeAndSourceLocation componentInfo = findTypeAndSourceLocation(c.get(), includeChildren);
                info.set(componentInfo);
            }
        });
        ComponentTypeAndSourceLocation res = info.get();
        if (res == null) {
            throw new IllegalArgumentException("Unable to find component in source");
        }
        return res;
    }

    /**
     * Finds the source location of a component.
     *
     * @param component
     *            the component to find
     * @param includeChildren
     *            whether to include children in the search
     * @return the source location of the component
     */
    public ComponentTypeAndSourceLocation findTypeAndSourceLocation(Component component, boolean includeChildren) {
        Optional<Component> parent = component.getParent();

        ComponentTypeAndSourceLocation parentInfo = parent.map(this::_getSourceLocation).orElse(null);

        List<ComponentTypeAndSourceLocation> children = new ArrayList<>();
        if (includeChildren) {
            component.getChildren().forEach(child -> {
                ComponentTypeAndSourceLocation childInfo = findTypeAndSourceLocation(child, true);
                children.add(childInfo);
            });
        }
        return getSourceLocation(component, parentInfo, children);
    }

    /**
     * Finds the {@link ComponentTypeAndSourceLocation} for given component with
     * <code>nodeId</code> in ui with id <code>uiId</code> by analyzing the source
     * code without using the createLocation provided by Flow.
     * <p>
     * Note: This method should be used if an operation requires source file instead
     * of creation location for the component
     * </p>
     *
     * @param nodeId
     *            Flow component node id
     * @param uiId
     *            Flow UI id
     * @return ComponentTypeAndSourceLocation if component class analyzed
     *         successfully, empty when file or component is not accessible.
     * @throws IOException
     *             thrown when file operation fails
     */
    public Optional<ComponentTypeAndSourceLocation> analyzeSourceFileAndGetComponentTypeAndSourceLocation(int nodeId,
            int uiId) throws IOException {
        Optional<Component> componentOptional = FlowUtil.findComponentByNodeIdAndUiId(vaadinSession, nodeId, uiId);
        if (componentOptional.isEmpty()) {
            return Optional.empty();
        }
        return analyzeSourceFileAndGetComponentTypeAndSourceLocation(componentOptional.get());
    }

    /**
     * Finds the {@link ComponentTypeAndSourceLocation} for given component with by
     * analyzing the source code without using the createLocation provided by Flow.
     * <p>
     * Note: This method should be used if an operation requires source file instead
     * of creation location for the component.
     * </p>
     * <p>
     * Note: This method does not include children and parent info.
     * </p>
     *
     * @param component
     *            Component instance
     * @return {@link ComponentTypeAndSourceLocation} if component class analyzed
     *         successfully, empty when file or component is not accessible.
     * @throws IOException
     *             thrown when file operation fails
     */
    public Optional<ComponentTypeAndSourceLocation> analyzeSourceFileAndGetComponentTypeAndSourceLocation(
            Component component) throws IOException {
        String className = component.getClass().getName();
        File fileForClass = getProjectFileManager().getFileForClass(className);
        if (!fileForClass.exists()) {
            return Optional.empty();
        }
        KotlinUtil.throwIfKotlin(fileForClass);
        ClassSourceStaticAnalyzer classSourceStaticAnalyzer = new ClassSourceStaticAnalyzer(this, component,
                fileForClass);
        ComponentTracker.Location createLocation = classSourceStaticAnalyzer
                .findCreateLocationCandidateOrThrow(vaadinSession);

        throwIfCorrupt(createLocation);

        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = new ComponentTypeAndSourceLocation(
                component.getClass(), ComponentSourceFinder.findInheritanceChain(component), component,
                Optional.of(fileForClass), Optional.of(createLocation), Optional.empty(), null, new ArrayList<>());
        return Optional.of(componentTypeAndSourceLocation);

    }

    private record LocationAndFile(ComponentTracker.Location location, File file) {
    }

    /**
     * Gets the source location for the given component.
     * <p>
     * NOTE that the session must be locked when calling this method.
     *
     * @return the source location of the component
     */
    public ComponentTypeAndSourceLocation _getSourceLocation(Component component) {
        return getSourceLocation(component, null, null);
    }

    private ComponentTypeAndSourceLocation getSourceLocation(Component component,
            ComponentTypeAndSourceLocation parentInfo, List<ComponentTypeAndSourceLocation> children) {
        ComponentTracker.Location[] createLocations = ComponentTracker.findCreateLocations(component);
        ComponentTracker.Location[] attachLocations = ComponentTracker.findAttachLocations(component);
        Optional<LocationAndFile> create = findProjectLocation(component, createLocations, Type.CREATE);
        Optional<LocationAndFile> attach = findProjectLocation(component, attachLocations, Type.ATTACH);
        Optional<ComponentTracker.Location> createLocation = create.map(LocationAndFile::location);
        Optional<ComponentTracker.Location> attachLocation = attach.map(LocationAndFile::location);
        Optional<File> javaFile = createLocation.map(getProjectFileManager()::getSourceFile);

        if (javaFile.isEmpty()) {
            // The component was not created in the project source
            createLocation = Optional.empty();
            attachLocation = Optional.empty();
        }

        createLocation.ifPresent(this::throwIfCorrupt);
        attachLocation.ifPresent(this::throwIfCorrupt);

        return new ComponentTypeAndSourceLocation(component.getClass(), findInheritanceChain(component), component,
                javaFile, createLocation, attachLocation, parentInfo, children);
    }

    private void throwIfCorrupt(ComponentTracker.Location location) {
        if (location.filename() == null && location.lineNumber() == -1) {
            getLogger().error(
                    "The source reference for the class {} is missing or corrupt. Recompile the class or restart the application.",
                    location.className());
            throw new SuggestRestartException("The source reference for the class " + location.className()
                    + " is corrupt. Recompile the class or restart the application.");
        }
    }

    private void throwIfFileRemoved(Component component) {

    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(ComponentSourceFinder.class);
    }

    private ProjectFileManager getProjectFileManager() {
        return ProjectFileManager.get();
    }

    /**
     * Finds inheritance chain of a component by going up for each superclass
     *
     * @param component
     *            Flow component
     * @return list of classes
     */
    public static List<Class<?>> findInheritanceChain(Component component) {
        List<Class<?>> inheritanceChain = new ArrayList<>();
        Class<?> superClass = component.getClass().getSuperclass();
        inheritanceChain.add(component.getClass());
        while (superClass != null) {
            inheritanceChain.add(superClass);
            superClass = superClass.getSuperclass();
        }

        return inheritanceChain;
    }

    private Optional<LocationAndFile> findProjectLocation(Component component, ComponentTracker.Location[] allLocations,
            Type type) {
        if (allLocations == null) {
            return Optional.empty();
        }

        List<ComponentTracker.Location> locations = Stream.of(allLocations).filter(location -> {
            for (String prefixToSkip : prefixesToSkip) {
                if (location.className().startsWith(prefixToSkip)) {
                    return false;
                }
            }
            return true;
        }).toList();

        // First find the <init> for the given component
        for (int i = 0; i < locations.size(); i++) {
            ComponentTracker.Location location = locations.get(i);
            if (type == Type.CREATE && isConstructor(location, component)) {

                int createIndex = i;
                // Internal constructor calls are not interesting
                while (isConstructor(locations.get(createIndex + 1), component)) {
                    createIndex++;
                }

                // This is the constructor call for the component with are looking for
                // The next stack trace is the place where the component is created (unless
                // there are some intermediate proxy layers like with Spring)
                // However, if the "intermediate" layers contain a component creation, we cannot
                // look at those as we would pick up the location where the parent component is
                // created instead
                Optional<LocationAndFile> componentCreate = findInProject(locations.get(createIndex + 1));

                if (componentCreate.isPresent()) {
                    return componentCreate;
                }
                // This could be a route class, then we return the init of the class itself
                return findInProject(locations.get(i));
            } else if (type == Type.ATTACH) {
                Optional<LocationAndFile> locationAndFileInProject = findInProject(location);
                if (locationAndFileInProject.isPresent()) {
                    return locationAndFileInProject;
                }
            }
        }

        return Optional.empty();
    }

    private static boolean isConstructor(ComponentTracker.Location location, Component component) {
        return location.className().equals(component.getClass().getName()) && "<init>".equals(location.methodName());
    }

    private Optional<LocationAndFile> findInProject(ComponentTracker.Location location) {
        File file = ProjectFileManager.get().getSourceFile(location);
        if (file.exists()) {
            return Optional.of(new LocationAndFile(location, file));
        }
        return Optional.empty();
    }
}
