package com.vaadin.copilot.startup;

import java.io.File;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import com.vaadin.flow.server.frontend.Options;
import com.vaadin.flow.server.frontend.TypeScriptBootstrapModifier;
import com.vaadin.flow.server.frontend.scanner.FrontendDependenciesScanner;
import com.vaadin.flow.theme.ThemeDefinition;

public class CopilotLoader implements TypeScriptBootstrapModifier {

    private String themeImportFolder;

    @Override
    public void modify(List<String> bootstrapTypeScript, Options options,
            FrontendDependenciesScanner frontendDependenciesScanner) {
        if (options.isProductionMode()) {
            return;
        }

        ThemeDefinition themeDefinition = frontendDependenciesScanner.getThemeDefinition();
        if (themeDefinition != null && themeDefinition.getTheme() != null
                && themeDefinition.getTheme().getName().equals("com.vaadin.flow.theme.material")) {
            this.themeImportFolder = "material";
        } else {
            this.themeImportFolder = "lumo";
        }

        String connectClientPath = getConnectClientPath(options);
        if (connectClientPath != null) {
            bootstrapTypeScript.addAll(0,
                    List.of(String.format(
                            """
                                    import client from '%s';
                                    let generatedId = 0;
                                    const copilotMiddleware: any = async function(
                                      context: any,
                                      next: any
                                    ) {
                                        const id = ++generatedId;
                                        const requestData = { endpoint: context.endpoint, method: context.method, params: context.params, id };
                                        (window as any).Vaadin.copilot.eventbus.emit('endpoint-request', requestData);

                                        const response: Response = await next(context);
                                        const status = response.status;
                                        const text = await response.clone().text();
                                        const responseData = { text, id, status };
                                        (window as any).Vaadin.copilot.eventbus.emit('endpoint-response', responseData);
                                        return response;
                                    };

                                    client.middlewares = [...client.middlewares, copilotMiddleware];
                                    """,
                            connectClientPath)));
        }

        if (options.isReactEnabled() && hasNpmModule(options, "react-router")) {
            bootstrapTypeScript.add("""
                    import { Outlet } from 'react-router';
                    (window as any).Vaadin ??= {};
                    (window as any).Vaadin.copilot ??= {};
                    (window as any).Vaadin.copilot._ref ??= {};
                    (window as any).Vaadin.copilot._ref.Outlet = Outlet;
                    """);
        }
        List<String> addAtTopList = new ArrayList<>();
        addAtTopList.addAll(List.of("import 'Frontend/generated/jar-resources/copilot.js';", """
                // @ts-ignore
                if (import.meta.hot) {
                  // @ts-ignore
                  import.meta.hot.on('vite:beforeUpdate', (e:any) => {
                    if ((window as any).Vaadin.copilot?.disableViteHmr) {
                        e.updates = [];
                    }
                  });
                  // @ts-ignore
                  import.meta.hot.on('vite:beforeFullReload', (payload:any) => {
                    if ((window as any).Vaadin.copilot?.disableViteHmr) {
                        payload.path = "something-not-used-in-the-app-to-prevent-reload.html";
                    }
                  });
                  // @ts-ignore
                  import.meta.hot.on('vite:afterUpdate', () => {
                    const eventbus = (window as any).Vaadin.copilot.eventbus;
                    if (eventbus) {
                      eventbus.emit('vite-after-update',{});
                    }
                  });
                }
                """));
        if (!options.isNpmExcludeWebComponents()) {
            addAtTopList.addAll(List.of(
                    // Imports for all the vaadin components used in copilot
                    // must go
                    // into the application bundle to avoid loading them
                    // twice.
                    // We must load the version that the application wants
                    // to use to
                    // avoid theme conflicts until
                    // https://github.com/vaadin/web-components/issues/7055
                    // is
                    // fixed.
                    themedImport("vertical-layout"), //
                    themedImport("horizontal-layout"), //
                    themedImport("context-menu"), //
                    themedImport("checkbox"), //
                    themedImport("text-field"), //
                    themedImport("text-area"), //
                    themedImport("menu-bar"), //
                    themedImport("grid"), //
                    themedImport("grid", "vaadin-grid-tree-column.js"), //
                    themedImport("details"), //
                    themedImport("select"), //
                    themedImport("overlay"), //
                    themedImport("list-box"), //
                    themedImport("combo-box"), //
                    themedImport("item"), //
                    themedImport("tabsheet"), //
                    themedImport("dialog"), //
                    themedImport("multi-select-combo-box"), //
                    themedImport("radio-group"), //
                    "import '@vaadin/icons/vaadin-iconset.js';", //
                    "import '@vaadin/icon/vaadin-icon.js';"));
        }
        bootstrapTypeScript.addAll(0, addAtTopList);
    }

    private boolean hasNpmModule(Options options, String... moduleName) {
        Path modulePath = options.getNpmFolder().toPath().resolve(Path.of("node_modules", moduleName));

        return (modulePath.toFile().exists());
    }

    private static String getConnectClientPath(Options options) {
        File customConnectClient = new File(options.getFrontendDirectory(), "connect-client.ts");
        File defaultConnectClient = new File(options.getFrontendGeneratedFolder(), "connect-client.default.ts");
        String connectClientPath = null;
        if (customConnectClient.exists()) {
            connectClientPath = "Frontend/connect-client.js";
        } else if (defaultConnectClient.exists()) {
            connectClientPath = "Frontend/generated/connect-client.default.js";
        }
        return connectClientPath;
    }

    private String themedImport(String component) {
        return themedImport(component, "vaadin-" + component + ".js");
    }

    private String themedImport(String component, String filename) {
        return "import '@vaadin/{component}/theme/{theme}/{filename}';".replace("{component}", component)
                .replace("{theme}", themeImportFolder).replace("{filename}", filename);
    }
}
