package com.vaadin.copilot;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.VaadinSession;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Acts as a bridge between Copilot and Spring specific API. Can be imported
 * into copilot and must never itself import Spring classes.
 */
public class SpringBridge {

    public record ServiceMethodInfo(Class<?> serviceClass, Method serviceMethod) {
    }

    public record VersionInfo(String springBootVersion, String springVersion) {
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(SpringSecurityIntegration.class);
    }

    private static boolean hasClass(String className) {
        try {
            Class.forName(className);
            return true;
        } catch (Throwable t) {
            return false;
        }

    }

    /**
     * Returns whether Spring is available in the classpath and enabled for the
     * given context.
     *
     * @param context
     *            the Vaadin context
     * @return true if Spring is available and enabled, false otherwise
     */
    public static boolean isSpringAvailable(VaadinServletContext context) {
        if (hasClass("org.springframework.web.context.WebApplicationContext")) {
            try {
                callSpring("getWebApplicationContext", context);
                return true;
            } catch (Exception e) {
                getLogger().trace("Failed to call getWebApplicationContext", e);
            }
        }
        return false;
    }

    /**
     * Returns whether Spring is available in the classpath.
     *
     * @return true if Spring is available, false otherwise
     */
    public static boolean isSpringSecurityAvailable() {
        return hasClass("org.springframework.security.config.annotation.web.configuration.EnableWebSecurity");
    }

    /**
     * Returns whether Spring Data JPA is available in the classpath.
     *
     * @return true if Spring Data JPA is available, false otherwise
     */
    public static boolean isSpringDataJpaAvailable(VaadinServletContext context) {
        return hasClass("org.springframework.data.jpa.repository.JpaSpecificationExecutor");
    }

    /**
     * Returns the value of the given property from the Spring environment of the
     * given context.
     *
     * @param context
     *            the Vaadin servlet context
     * @param property
     *            the property name
     * @return the property value or null if not found
     */
    public static String getPropertyValue(VaadinServletContext context, String property) {
        return (String) callSpring("getPropertyValue", context, property);
    }

    private static Object callSpring(String methodName, Object... parameters) {
        return call("com.vaadin.copilot.SpringIntegration", methodName, parameters);
    }

    private static Object callSpringSecurity(String methodName, Object... parameters) {
        return call("com.vaadin.copilot.SpringSecurityIntegration", methodName, parameters);
    }

    private static Object callSpringData(String methodName, Object... parameters) {
        return call("com.vaadin.copilot.SpringDataIntegration", methodName, parameters);
    }

    private static Object call(String className, String methodName, Object... parameters) {
        try {
            Method method = Arrays.stream(Class.forName(className).getMethods()).filter(m -> {
                if (!m.getName().equals(methodName)) {
                    return false;
                }
                return parameters == null || parameters.length == m.getParameterCount();
            }).findFirst().orElseThrow(NoSuchMethodException::new);
            return method.invoke(null, parameters);
        } catch (NoSuchMethodException | ClassNotFoundException | IllegalAccessException e) {
            throw new CopilotException(e);
        } catch (InvocationTargetException e) {
            Throwable targetException = e.getTargetException();
            if (targetException instanceof RuntimeException) {
                throw (RuntimeException) targetException;
            }
            throw new CopilotException(targetException);
        }
    }

    /**
     * Returns the Spring Boot application class of the given context.
     *
     * @param context
     *            the Vaadin servlet context
     * @return the Spring Boot application class or null if not found
     */
    public static Class<?> getApplicationClass(VaadinServletContext context) {
        return (Class<?>) callSpring("getApplicationClass", context);
    }

    /**
     * Returns the Spring Boot application class name of the given context.
     * <p>
     * Mostly exists for tests where the class instance does not exist.
     * 
     * @param context
     *            the Vaadin servlet context
     * @return the Spring Boot application class or null if not found
     */
    public static String getApplicationClassName(VaadinServletContext context) {
        return getApplicationClass(context).getName();
    }

    /**
     * Returns whether Spring Security is enabled in the given context.
     *
     * @param context
     *            the Vaadin servlet context
     * @return true if Spring Security is enabled, false otherwise
     */
    public static boolean isViewSecurityEnabled(VaadinServletContext context) {
        return callSpring("isViewSecurityEnabled", context) == Boolean.TRUE;
    }

    /**
     * Gets the prefix used for all route paths in the application
     *
     * @param context
     *            the Vaadin servlet context
     * @return the prefix used for all route paths in the application, never ending
     *         in a "/" or an empty string if no prefix is used
     */
    public static String getUrlPrefix(VaadinServletContext context) {
        String value = getPropertyValue(context, "vaadin.url-mapping");
        if (value == null) {
            return "";
        }
        if (value.endsWith("/*")) {
            value = value.substring(0, value.length() - 2);
        }
        if (value.endsWith("/")) {
            value = value.substring(0, value.length() - 1);
        }
        return value;
    }

    /**
     * Gets a list of all endpoints / browser callables in the application.
     *
     * @return a list of endpoint info objects
     */
    public static List<ServiceMethodInfo> getEndpoints(VaadinServletContext context) {
        return (List<ServiceMethodInfo>) callSpring("getEndpoints", context);
    }

    /**
     * Gets a list of @Service classes that can potentially be used to get data to a
     * component.
     *
     * @return a list service classes
     */
    public static List<ServiceMethodInfo> getFlowUIServices(VaadinServletContext context) {
        return (List<ServiceMethodInfo>) callSpring("getFlowUIServices", context);
    }

    /**
     * Gets version information for Spring Boot and related libraries.
     *
     * @return version information
     */
    public static VersionInfo getVersionInfo() {
        return (VersionInfo) callSpring("getVersionInfo");
    }

    /**
     * Checks whether Spring Security is enabled.
     *
     * @param context
     *            the Vaadin servlet context
     * @return true if Spring Security is enabled, false otherwise
     */
    public static boolean isSpringSecurityEnabled(VaadinServletContext context) {
        if (!isSpringSecurityAvailable()) {
            return false;
        }
        return (boolean) callSpringSecurity("isSpringSecurityEnabled", context);
    }

    /**
     * Set active user for Spring Security.
     *
     * @param username
     *            the username
     * @param session
     *            the Vaadin session
     */
    public static void setActiveSpringSecurityUser(String username, VaadinSession session) {
        callSpringSecurity("setActiveSpringSecurityUser", username, session);
    }

    /**
     * Lists the JPA entities in the project
     *
     * @param context
     *            the Vaadin context
     * @return a list of JPA entity classes in the project
     */
    public static List<Class<?>> getJpaEntityClasses(VaadinServletContext context) {
        if (!isSpringDataJpaAvailable(context)) {
            return List.of();
        }
        return (List<Class<?>>) callSpringData("getJpaEntityClasses", context);
    }

    public record H2Info(String jdbcUrl, String path) {
    }

    /**
     * Gets information about a H2 database.
     *
     * @param context
     *            the Vaadin context
     * @return an optional H2Info object containing information about the H2
     *         database, or empty if no H2 database is in use
     */
    public static Optional<H2Info> getH2Info(VaadinServletContext context) {
        return (Optional<H2Info>) callSpring("getH2Info", context);
    }

}
