package com.vaadin.copilot;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.pro.licensechecker.MachineId;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Copilot 3 day preview handler. It has two basic functionality that starts a
 * new preview and get the preview state.
 */
public class PreviewHandler extends CopilotCommand {
    private static final String REMAINING_TIME_IN_MILLIS = "remainingTimeInMillis";
    private static final String PREVIEW_ACTIVATED = "previewActivated";
    private static final String PREVIEW_ACTIVE = "active";
    private static volatile ActivationResponse latestPreview;

    private final CopilotServerClient copilotServerClient = new CopilotServerClient();

    record PreviewRequest(String machineId) {
    }

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (!(command.equals("start-preview") || command.equals("get-preview"))) {
            return false;
        }

        if (!data.has(KEY_REQ_ID)) {
            getLogger().warn("Missing required parameter 'reqId' in message: {}", data);
            devToolsInterface.send(Copilot.PREFIX + command + "-response", JacksonUtils.createObjectNode());
            return true;
        }
        ObjectNode responseData = JacksonUtils.createObjectNode();
        responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asText());

        if (command.equals("start-preview")) {
            try {
                startPreview();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            } catch (Exception e) {
                getLogger().debug("Could not start preview", e);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData,
                        "Could not connect server for starting the trial", e);
                return true;
            }
        }
        putPreviewData(responseData);
        devToolsInterface.send("copilot-get-preview-response", responseData);
        return true;
    }

    private void startPreview() throws IOException, InterruptedException {
        URI uri = copilotServerClient.getQueryURI("activation");
        HttpRequest request = copilotServerClient.buildRequest(uri, new PreviewRequest(MachineId.get()));
        HttpResponse<String> response = copilotServerClient.getHttpClient().send(request,
                HttpResponse.BodyHandlers.ofString());
        if (response.statusCode() != 200) {
            throw new IllegalArgumentException("Starting preview failed");
        }
        ActivationResponse previewStartResponse = CopilotJacksonUtils.readValue(response.body(),
                ActivationResponse.class);
        if (previewStartResponse == null || !previewStartResponse.active) {
            throw new IllegalArgumentException("Starting preview failed");
        }
    }

    private void putPreviewData(ObjectNode responseJson) {

        responseJson.put(PREVIEW_ACTIVATED, false);
        try {
            URI uri = copilotServerClient.getQueryURI("activation?machineId=" + MachineId.get());
            HttpRequest request = HttpRequest.newBuilder().uri(uri).GET().timeout(Duration.ofSeconds(120)).build();
            HttpResponse<String> response = copilotServerClient.getHttpClient().send(request,
                    HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() == 404) {
                responseJson.put(PREVIEW_ACTIVATED, false);
                responseJson.put(PREVIEW_ACTIVE, false);
                return;
            }
            ActivationResponse getPreviewResponse = CopilotJacksonUtils.readValue(response.body(),
                    ActivationResponse.class);

            latestPreview = getPreviewResponse;

            responseJson.put(PREVIEW_ACTIVE, getPreviewResponse.active);
            responseJson.put(PREVIEW_ACTIVATED, true);
            responseJson.put(REMAINING_TIME_IN_MILLIS, getPreviewResponse.timeUntilExpireInMillis());
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        } catch (Exception ex) {
            // silent exception
            getLogger().debug("Could not read preview state", ex);
        }
    }

    /**
     * Returns the latest preview activation response.
     *
     * @return ActivationResponse - the latest preview activation response
     */
    public static ActivationResponse getLatestPreview() {
        return latestPreview;
    }

    public record ActivationResponse(boolean active, long activationTimeInMillis, long timeUntilExpireInMillis) {
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(PreviewHandler.class);
    }
}
