package com.vaadin.copilot;

import com.vaadin.copilot.analytics.AnalyticsClient;
import com.vaadin.flow.server.ServiceInitEvent;
import com.vaadin.flow.server.VaadinServiceInitListener;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

import org.jsoup.nodes.Document;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CopilotIndexHtmlLoader implements VaadinServiceInitListener {

    // Suppress duplicate lit version messages as we always cause that
    // Copilot is always in development mode
    // The deprecation warnings are caused by the polylit mixin, no
    // need for everybody to see them
    private final String ignoreLitWarnings = """
            const issuedWarnings = window.litIssuedWarnings ??= new Set();
            issuedWarnings.add(
              'Multiple versions of Lit loaded. Loading multiple versions is not recommended. See https://lit.dev/msg/multiple-versions for more information.',
            );
            issuedWarnings.add(
              'Lit is in dev mode. Not recommended for production! See https://lit.dev/msg/dev-mode for more information.',
            );
            issuedWarnings.add(
              'Overriding ReactiveElement.createProperty() is deprecated. The override will not be called with standard decorators See https://lit.dev/msg/no-override-create-property for more information.',
            );
            issuedWarnings.add(
              'Overriding ReactiveElement.getPropertyDescriptor() is deprecated. The override will not be called with standard decorators See https://lit.dev/msg/no-override-get-property-descriptor for more information.',
            );
            """;

    @Override
    public void serviceInit(ServiceInitEvent event) {
        VaadinServletContext context = (VaadinServletContext) event.getSource().getContext();
        if (!CopilotStatus.isEnabled(context)) {
            getLogger().info("Copilot is disabled");
            return;
        }

        boolean vaadinUsageStatisticsEnabled = ApplicationConfiguration.get(context).isUsageStatisticsEnabled();
        boolean analyticsEnabled = !Copilot.isDevelopmentMode() && vaadinUsageStatisticsEnabled;
        AnalyticsClient.setEnabled(analyticsEnabled);

        event.addIndexHtmlRequestListener(e -> {
            Document document = e.getDocument();
            String devToolsConf = getRow(document.head().toString(), "window.Vaadin.devToolsConf");
            if (devToolsConf.contains("\"token\":")) {
                // Only add copilot if a debug connection is allowed. This is
                // not a security feature but prevents showing a non-functional
                // copilot
                document.body().append("<copilot-main></copilot-main>");
                document.head().appendElement("script").append(ignoreLitWarnings);
            }
        });
    }

    private String getRow(String text, String needle) {
        int location = text.indexOf(needle);
        if (location == -1) {
            return "";
        }
        int prependingNewline = text.substring(0, location).lastIndexOf("\n");
        int followingNewline = text.indexOf("\n", location);
        return text.substring(prependingNewline + 1, followingNewline);
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
