package com.vaadin.copilot.javarewriter;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.vaadin.flow.dom.Style;
import com.vaadin.flow.shared.util.SharedUtil;

import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.stmt.ExpressionStmt;

public class JavaStyleRewriter {

    public record StyleInfo(String property, String value, boolean isDashSeparatedProperty) {
    }

    /**
     * Gets the (active) styles of a component.
     *
     * @param componentInfo
     *            the component to get the styles of
     * @return the styles, as a list of style names and values
     */
    public static List<StyleInfo> getStyles(ComponentInfo componentInfo) {
        return JavaRewriterUtil.findCalls(Style.class, componentInfo).stream().map(JavaStyleRewriter::extractStyle)
                .toList();
    }

    /**
     * Sets the given inline style on the given component, replacing an existing
     * style property if present.
     *
     * @param componentInfo
     *            the component to set the style on
     * @param dashSeparatedProperty
     *            the style property to set
     * @param value
     *            the style value to set or null to remove the style
     */
    public static void setStyle(ComponentInfo componentInfo, String dashSeparatedProperty, String value) {
        List<MethodCallExpr> styleCalls = JavaRewriterUtil.findCalls(Style.class, componentInfo);
        List<MethodCallExpr> existingSetters = getExistingSetterMethodCalls(componentInfo, dashSeparatedProperty);
        if (existingSetters.isEmpty()) {
            if (value == null) {
                throw new IllegalArgumentException(
                        "Unable to remove non-existing style " + dashSeparatedProperty + " from " + componentInfo);
            }
            MethodCallExpr newStylesCall = createStylesCall(dashSeparatedProperty, value);

            if (!styleCalls.isEmpty()) {
                // Add a new style call by chaining to the last existing
                // style call
                MethodCallExpr lastStylesCall = styleCalls.get(styleCalls.size() - 1);

                JavaRewriterUtil.findAncestorOrThrow(lastStylesCall, ExpressionStmt.class).setExpression(newStylesCall);
                newStylesCall.setScope(lastStylesCall);
            } else {
                // Add a new getStyle() call and use that to set the style
                MethodCallExpr getStyleCall = JavaRewriterUtil.addFunctionCall(componentInfo, "getStyle",
                        Collections.emptyList());
                newStylesCall.setScope(JavaRewriterUtil.clone(getStyleCall));
                getStyleCall.replace(newStylesCall);
            }

        } else {
            // Replace the value in the last existing setter
            updateStyle(existingSetters.get(existingSetters.size() - 1), value);
        }
    }

    private static List<MethodCallExpr> getExistingSetterMethodCalls(ComponentInfo componentInfo,
            String dashSeparatedProperty) {
        List<MethodCallExpr> styleCalls = JavaRewriterUtil.findCalls(Style.class, componentInfo);
        return styleCalls.stream().filter(methodCallExpr -> {
            StyleInfo style = extractStyle(methodCallExpr);
            if (style.isDashSeparatedProperty) {
                return style.property().equals(dashSeparatedProperty);
            } else {
                return SharedUtil.camelCaseToDashSeparated(style.property()).equals(dashSeparatedProperty);
            }
        }).toList();
    }

    private static MethodCallExpr createStylesCall(String dashSeparatedProperty, String value) {
        String camelCaseProperty = SharedUtil.dashSeparatedToCamelCase(dashSeparatedProperty);
        String setter = JavaRewriterUtil.getSetterName(camelCaseProperty, Style.class, false);
        if (JavaRewriterUtil.hasSetterForType(Style.class, setter, String.class)) {
            return new MethodCallExpr(null, setter).addArgument(JavaRewriterUtil.toExpression(value));
        }
        return new MethodCallExpr(null, "set").addArgument(JavaRewriterUtil.toExpression(dashSeparatedProperty))
                .addArgument(JavaRewriterUtil.toExpression(value));
    }

    /**
     * Extracts the style information from a method call to a method inside Style.
     *
     * <p>
     * Note that this only processes the method call and not any chained calls.
     *
     * @param methodCallExpr
     *            a method call that refers to a method inside Style.
     * @return The style info for the method call.
     */
    private static StyleInfo extractStyle(MethodCallExpr methodCallExpr) {
        String setter = methodCallExpr.getNameAsString();
        if (setter.equals("set")) {
            String dashProperty = String.valueOf(JavaRewriterUtil.fromExpression(methodCallExpr.getArgument(0), null));
            String value = String.valueOf(JavaRewriterUtil.fromExpression(methodCallExpr.getArgument(1), null));
            return new StyleInfo(dashProperty, value, true);
        } else {
            int argCount = methodCallExpr.getArguments().size();
            if (argCount != 1) {
                throw new IllegalArgumentException(
                        "Expected styles method call expression to have one argument but was " + argCount + " for "
                                + methodCallExpr);
            }
            Expression argument = methodCallExpr.getArgument(0);
            String value;
            if (argument.isFieldAccessExpr()) {
                String fieldName = argument.asFieldAccessExpr().getNameAsString();
                // This is copied from Styles.java
                value = fieldName.replace("_", "-").toLowerCase();
            } else if (argument.isStringLiteralExpr()) {
                value = argument.asStringLiteralExpr().getValue();
            } else {
                throw new IllegalArgumentException("Unexpected argument type in style call: " + methodCallExpr);
            }
            String property = JavaRewriterUtil.getPropertyName(setter);
            return new StyleInfo(property, value, false);
        }
    }

    static void updateStyle(MethodCallExpr methodCallExpr, Object value) {
        if (value == null) {
            JavaRewriterUtil.removeFromChainedStyleCall(methodCallExpr);
            return;
        }
        if (methodCallExpr.getNameAsString().equals("set")) {
            methodCallExpr.getArgument(1).replace(JavaRewriterUtil.toExpression(value));
        } else {
            methodCallExpr.getArgument(0).replace(JavaRewriterUtil.toExpression(value));
        }
    }

    /**
     * Handles calls of the given component and adjust them based on the given
     * properties.
     *
     * @param componentInfo
     *            Component to update
     * @param changes
     *            Changed properties for the sizing. Keys are camelCased, values
     *            might be string or null. e.g. <code>flexGrow -> "1"</code>,
     *            <code>maxHeight -> null</code>
     */
    public static void setSizing(ComponentInfo componentInfo, Map<String, String> changes) {
        List<MethodCallExpr> methodCalls = JavaRewriterUtil.findMethodCalls(componentInfo);
        changes.forEach((key, value) -> {
            if (value == null) {
                removeSizingStatements(componentInfo, methodCalls, key);
            } else {
                setStyle(componentInfo, SharedUtil.camelCaseToDashSeparated(key), value);
            }
        });
    }

    private static void removeSizingStatements(ComponentInfo componentInfo, List<MethodCallExpr> methodCalls,
            String camelCasePropertyKey) {
        boolean propertyRemoved = false;
        String setterName = JavaRewriterUtil.getSetterName(camelCasePropertyKey, componentInfo.getClass(), false);
        // setWidth or setHeight
        List<MethodCallExpr> list = methodCalls.stream()
                .filter(methodCallExpr -> methodCallExpr.getNameAsString().equals(setterName)).toList();
        if (!list.isEmpty()) {
            propertyRemoved = true;
            list.forEach(JavaRewriterUtil::removeStatement);
        }
        if (!propertyRemoved) {
            propertyRemoved = removeExpandStatements(componentInfo, camelCasePropertyKey);
        }
        if (!propertyRemoved) {
            setStyle(componentInfo, SharedUtil.camelCaseToDashSeparated(camelCasePropertyKey), null);
        }
    }

    private static boolean removeExpandStatements(ComponentInfo componentInfo, String camelCasePropertyKey) {
        if (!"flexGrow".equals(camelCasePropertyKey)) {
            return false;
        }
        List<Expression> parameterUsage = JavaRewriterUtil.findParameterUsage(componentInfo).stream()
                .filter(parameterExpression -> parameterExpression.getParentNode().isPresent()
                        && parameterExpression.getParentNode().get() instanceof MethodCallExpr)
                .toList();
        for (Expression expression : parameterUsage) {
            Optional<Node> parentNode = expression.getParentNode();
            if (parentNode.isEmpty()) {
                throw new IllegalStateException("Did not expect parent node to be empty");
            }
            var methodCallExpr = (MethodCallExpr) parentNode.get();
            if (methodCallExpr.getNameAsString().equals("expand")) {
                if (methodCallExpr.getArguments().size() == 1) {
                    JavaRewriterUtil.removeStatement(methodCallExpr);
                    return true;
                } else {
                    int argumentPosition = methodCallExpr.getArgumentPosition(expression);
                    methodCallExpr.getArguments().remove(argumentPosition);
                }
            } else if (methodCallExpr.getNameAsString().equals("addAndExpand")) {
                throw new IllegalArgumentException("addAndExpand call is not supported yet");
            }
        }
        return false;
    }
}
