package com.vaadin.copilot;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.auth.AnnotatedViewAccessChecker;
import com.vaadin.flow.server.auth.NavigationAccessControl;
import com.vaadin.hilla.EndpointRegistry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.cglib.proxy.Proxy;
import org.springframework.core.env.Environment;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

/**
 * Provides Spring related helpers for copilot. Depends on Spring classes and
 * cannot be directly imported
 */
public class SpringIntegration {

    /**
     * Returns the value of the given property from the Spring environment of the
     * given context. See {@link SpringBridge}
     *
     * @param context
     *            the Vaadin servlet context
     * @param property
     *            the property name
     * @return the property value or null if not found
     */
    public static String getPropertyValue(VaadinServletContext context, String property) {
        WebApplicationContext webAppContext = WebApplicationContextUtils.getWebApplicationContext(context.getContext());
        if (webAppContext == null) {
            getLogger().error("No web application context available");
            return null;
        }
        Environment env = webAppContext.getEnvironment();
        return env.getProperty(property, (String) null);
    }

    /**
     * Returns the Spring Boot application class of the given context. See
     * {@link SpringBridge}
     *
     * @param context
     *            the Vaadin servlet context
     * @return the Spring Boot application class or null if not found
     */
    public static Class<?> getApplicationClass(VaadinServletContext context) {
        WebApplicationContext webAppContext = WebApplicationContextUtils.getWebApplicationContext(context.getContext());
        if (webAppContext == null) {
            getLogger().error("No web application context available");
            return null;
        }
        Map<String, Object> beans = webAppContext.getBeansWithAnnotation(SpringBootApplication.class);
        Class<?> appClass = beans.values().iterator().next().getClass();
        if (Proxy.isProxyClass(appClass)) {
            appClass = appClass.getSuperclass();
        }
        while (appClass.getName().contains("$$SpringCGLIB$$")) {
            appClass = appClass.getSuperclass();
        }
        return appClass;
    }

    /**
     * Returns whether Spring Security is enabled in the given context See
     * {@link SpringBridge}
     *
     * @param context
     *            the Vaadin servlet context
     * @return true if Spring Security is enabled, false otherwise
     */
    public static Boolean isViewSecurityEnabled(VaadinServletContext context) {
        WebApplicationContext webAppContext = WebApplicationContextUtils.getWebApplicationContext(context.getContext());
        if (webAppContext == null) {
            return false;
        }
        String[] naviAccessControl = webAppContext.getBeanNamesForType(NavigationAccessControl.class);
        if (naviAccessControl.length != 1) {
            return false;
        }
        NavigationAccessControl accessControl = (NavigationAccessControl) webAppContext.getBean(naviAccessControl[0]);
        return accessControl.hasAccessChecker(AnnotatedViewAccessChecker.class);
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(SpringIntegration.class);
    }

    public static List<SpringBridge.EndpointInfo> getEndpoints(VaadinServletContext context) {
        WebApplicationContext webAppContext = WebApplicationContextUtils.getWebApplicationContext(context.getContext());
        if (webAppContext == null) {
            return Collections.emptyList();
        }
        EndpointRegistry endpointRegistry = webAppContext.getBean(EndpointRegistry.class);
        Map<String, EndpointRegistry.VaadinEndpointData> vaadinEndpoints = endpointRegistry.getEndpoints();

        return getEndpointInfos(vaadinEndpoints);
    }

    private static List<SpringBridge.EndpointInfo> getEndpointInfos(
            Map<String, EndpointRegistry.VaadinEndpointData> vaadinEndpoints) {
        List<SpringBridge.EndpointInfo> endpointInfos = new ArrayList<>();
        for (EndpointRegistry.VaadinEndpointData endpoint : vaadinEndpoints.values()) {
            Map<String, Method> methods = endpoint.getMethods();
            for (Method method : methods.values()) {
                Class<?> endpointClass = endpoint.getEndpointObject().getClass();
                if (endpointClass.getName().startsWith("com.vaadin")) {
                    // Ignore internal endpoints
                    continue;
                }
                endpointInfos.add(new SpringBridge.EndpointInfo(endpointClass, method));
            }
        }
        return endpointInfos;
    }
}
