package com.vaadin.copilot.plugins.themeeditor.handlers;

import com.vaadin.copilot.plugins.themeeditor.ThemeEditorCommand;
import com.vaadin.copilot.plugins.themeeditor.messages.ClassNameRequest;
import com.vaadin.copilot.plugins.themeeditor.messages.ClassNameResponse;
import com.vaadin.copilot.plugins.themeeditor.utils.HasThemeModifier;
import com.vaadin.copilot.plugins.themeeditor.utils.MessageHandler;
import com.vaadin.flow.internal.JsonUtils;
import elemental.json.JsonObject;
import org.apache.commons.io.FilenameUtils;

import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Returns the class name of given component. If class name exists in theme
 * editor, it returns the existing one otherwise suggests a new one.
 */
public class ClassNameHandler implements MessageHandler {
    private static final Pattern SUGGEST_CLASS_NAME_PATTERN = Pattern
            .compile("(?!^)(?=[A-Z][a-z])");
    private final HasThemeModifier hasThemeModifier;

    public ClassNameHandler(HasThemeModifier hasThemeModifier) {
        this.hasThemeModifier = hasThemeModifier;
    }

    @Override
    public ExecuteAndUndo handle(JsonObject data) {
        List<String> existingClassNames = hasThemeModifier.getThemeModifier()
                .getExistingClassNames();
        ClassNameRequest request = JsonUtils.readToObject(data,
                ClassNameRequest.class);
        return findExistingClassName(existingClassNames, request)
                .orElseGet(() -> suggestClassName(existingClassNames, request));
    }

    private ExecuteAndUndo suggestClassName(List<String> existingClassNames,
            ClassNameRequest request) {
        String baseName = FilenameUtils.getBaseName(request.getFileName()) + "-"
                + request.getComponentDisplayName();
        String name = baseName;
        String selectorName = request.getComponentName() + "." + name;
        int i = 1;
        while (existingClassNames.contains(selectorName)) {
            name = baseName + "-" + i;
            selectorName = appendComponentTag(request.getComponentName(), name);
            i++;
        }
        String dashedName = dashify(name);
        return new ExecuteAndUndo(() -> new ClassNameResponse(dashedName, true),
                Optional.empty());
    }

    private Optional<ExecuteAndUndo> findExistingClassName(
            List<String> existingClassNames, ClassNameRequest request) {
        for (String componentClass : request.getClassList()) {
            if (existingClassNames.contains(appendComponentTag(
                    request.getComponentName(), componentClass))) {
                return Optional.of(new ExecuteAndUndo(
                        () -> new ClassNameResponse(componentClass, false),
                        Optional.empty()));
            }
        }
        return Optional.empty();
    }

    private String dashify(String text) {
        if (text == null) {
            return null;
        }
        final String subst = "-";
        final Matcher matcher = SUGGEST_CLASS_NAME_PATTERN.matcher(text);
        return matcher.replaceAll(subst).toLowerCase(Locale.ENGLISH)
                .replaceAll("-+", "-");

    }

    private String appendComponentTag(String tag, String str) {
        return tag + "." + str;
    }

    @Override
    public String getCommandName() {
        return ThemeEditorCommand.GET_CLASS_NAME.getValue();
    }
}
