package com.vaadin.copilot;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.pro.licensechecker.BuildType;
import com.vaadin.pro.licensechecker.LicenseChecker;
import com.vaadin.pro.licensechecker.LicenseException;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.ProKey;
import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Handles getting information about the user (pro key based) and communicating
 * with the server about feature access.
 */
public class UserInfoHandler implements CopilotCommand {

    public static final String PRODUCT_VAADIN_EMPLOYEE = "vaadin-employee";
    public static final String PRODUCT_VAADIN_COPILOT = "vaadin-copilot";

    @Override
    public boolean handleMessage(String command, JsonObject data,
            DevToolsInterface devToolsInterface) {
        if (command.equals("get-user-info")) {
            ProKey proKey = LocalProKey.get();
            String[] accessTo = {};
            if (proKey != null) {
                accessTo = queryAccessTo(proKey);
            }
            JsonObject userInfo = Json.createObject();
            if (proKey != null) {
                userInfo.put("proKey", proKey.getProKey());
            }

            JsonArray accessValues = Arrays.stream(accessTo).map(Json::create)
                    .collect(JsonUtils.asArray());
            userInfo.put("accessTo", accessValues);
            userInfo.put(KEY_REQ_ID, data.getString(KEY_REQ_ID));

            devToolsInterface.send("copilot-user-info", userInfo);
            return true;
        } else if (command.equals("log-in")) {
            ProKey proKey = LocalProKey.get();
            JsonObject response = Json.createObject();
            response.put(KEY_REQ_ID, data.getString(KEY_REQ_ID));
            if (proKey == null) {
                LicenseChecker.checkLicense(PRODUCT_VAADIN_COPILOT,
                        CopilotVersion.getVersion(), BuildType.DEVELOPMENT,
                        loginUrl -> {
                            // This is where we always should end up as there is
                            // no pro key
                            response.put("loginUrl", loginUrl);
                            devToolsInterface.send("log-in-resp", response);
                        });
            }
            return true;
        }
        return false;
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }

    private String[] queryAccessTo(ProKey proKey) {
        List<String> access = new ArrayList<>();
        try {
            LicenseChecker.checkLicense(PRODUCT_VAADIN_EMPLOYEE, "1.0.0",
                    BuildType.DEVELOPMENT, url -> {
                        // Do not ask the user to log in
                    });
            access.add(PRODUCT_VAADIN_EMPLOYEE);
        } catch (LicenseException e) {
            if (!ignoreWhenDoingBackgroundCheck(e)) {
                getLogger().error("Problem checking the sub status", e);
            }
        }
        try {
            LicenseChecker.checkLicense(PRODUCT_VAADIN_COPILOT,
                    CopilotVersion.getVersion(), BuildType.DEVELOPMENT, url -> {
                        // Do not ask the user to log in
                    });
            access.add(PRODUCT_VAADIN_COPILOT);
        } catch (LicenseException e) {
            if (!ignoreWhenDoingBackgroundCheck(e)) {
                getLogger().error("Problem checking the subscription status",
                        e);
            }
        }

        return access.toArray(String[]::new);
    }

    private boolean ignoreWhenDoingBackgroundCheck(LicenseException e) {
        String msg = e.getMessage();
        if (msg.contains("The provided license key does not allow usage")) {
            // There is a key but it does not allow access
            return true;
        }
        if (msg.contains("could not be reached and no offline key was found")) {
            // Offline
            return true;
        }

        return false;
    }
}
