package com.vaadin.copilot;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.base.devserver.DevToolsMessageHandler;
import com.vaadin.copilot.analytics.AnalyticsClient;
import com.vaadin.flow.server.VaadinService;
import com.vaadin.flow.server.startup.ApplicationConfiguration;
import elemental.json.JsonObject;

import java.io.IOException;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class Copilot implements DevToolsMessageHandler {

    static final String PREFIX = "copilot-";

    static final String VALUE = "value";

    static final String CONTEXT = "context";

    ProjectManager projectManager;

    private List<CopilotCommand> commands;

    private AnalyticsClient analyticsClient;

    @Override
    public void handleConnect(DevToolsInterface devToolsInterface) {
        try {
            ApplicationConfiguration applicationConfiguration = ApplicationConfiguration
                    .get(VaadinService.getCurrent().getContext());
            Path projectRoot = applicationConfiguration.getProjectFolder()
                    .toPath();
            projectManager = new ProjectManager(projectRoot);
            if (System.getProperty("copilot.development") == null
                    && isUsageStatisticsEnabled()) {
                analyticsClient = new AnalyticsClient();
            }
            setupCommands(devToolsInterface, applicationConfiguration);
            devToolsInterface.send(PREFIX + "init", null);
        } catch (IOException e) {
            throw new RuntimeException("Unable to create project manager", e);
        }
    }

    private void setupCommands(DevToolsInterface devToolsInterface,
            ApplicationConfiguration applicationConfiguration) {
        commands = List.of(new OpenComponentInIDE(applicationConfiguration),
                new ProjectFileHandler(projectManager),
                new AICommandHandler(projectManager), new UserInfoHandler());
    }

    @Override
    public boolean handleMessage(String command, JsonObject data,
            DevToolsInterface devToolsInterface) {
        if (!command.startsWith(PREFIX)) {
            return false;
        }
        Map<String, String> context = getContextMap(data);
        // remove context from data not to cause unmarshalling issues
        data.remove(CONTEXT);
        if (command.equals("copilot-track-event")) {
            String event = PREFIX + data.getString("event");
            if (data.hasKey(VALUE)) {
                track(event, Map.of(VALUE, data.getString(VALUE)), context);
            } else {
                track(event, null, context);
            }
            return true;
        }
        try {
            command = command.substring(PREFIX.length());
            for (CopilotCommand copilotCommand : commands) {
                if (copilotCommand.handleMessage(command, data,
                        devToolsInterface)) {
                    return true;
                }
            }
        } finally {
            track("copilot-command-" + command, null, context);
        }
        return false;
    }

    private void track(String event, Map<String, String> properties,
            Map<String, String> context) {
        if (analyticsClient != null) {
            analyticsClient.track(event, properties, context);
        }
    }

    private boolean isUsageStatisticsEnabled() {
        return ApplicationConfiguration
                .get(VaadinService.getCurrent().getContext())
                .isUsageStatisticsEnabled();
    }

    private Map<String, String> getContextMap(JsonObject data) {
        JsonObject context = data.getObject(CONTEXT);
        Map<String, String> contextMap = new HashMap<>();
        for (String key : context.keys()) {
            contextMap.put(key, context.getString(key));
        }
        return contextMap;
    }
}
