package com.unity3d.services.store

import com.unity3d.services.core.properties.ClientProperties
import com.unity3d.services.store.core.StoreEventListenerFactory
import com.unity3d.services.store.core.StoreExceptionHandler
import com.unity3d.services.store.core.StoreLifecycleListener
import com.unity3d.services.store.gpbl.StoreBilling
import kotlinx.coroutines.flow.MutableStateFlow
import java.lang.reflect.InvocationTargetException

class StoreMonitor(
    private val storeEventListenerFactory: StoreEventListenerFactory,
    private val storeExceptionHandler: StoreExceptionHandler
) {
    private var storeBilling: StoreBilling? = null
    private var storeLifecycleListener: StoreLifecycleListener? = null

    private val _isInitialized = MutableStateFlow(false)
    val isInitialized: Boolean
        get() = _isInitialized.value

	@Throws(
        ClassNotFoundException::class,
        NoSuchMethodException::class,
        IllegalAccessException::class,
        InvocationTargetException::class
    )
    fun initialize() {
        val storeEventListener = storeEventListenerFactory()
        if (isInitialized) {
            // Already initialized
            storeEventListener.triggerIsAlreadyInitialized()
            return
        }
        storeBilling = StoreBilling(ClientProperties.getApplicationContext(), storeEventListener)
        storeBilling?.initialize(storeEventListener)
        _isInitialized.value = true
    }

	fun isFeatureSupported(operationId: Int, purchaseType: String?): Int {
        val storeEventListener = storeEventListenerFactory(operationId)
        var result = -1
        try {
            result = storeBilling?.isFeatureSupported(purchaseType) ?: -1
            storeEventListener.triggerFeatureSupported(result)

        } catch (exception: Exception) {
            storeExceptionHandler.handleStoreException(
                StoreEvent.IS_FEATURE_SUPPORTED_REQUEST_ERROR,
                operationId,
                exception
            )
        }
        return result
    }

	fun getPurchases(operationId: Int, purchaseType: String?) {
        val storeEventListener = storeEventListenerFactory(operationId)
        try {
            checkNotNull(storeBilling)
            storeBilling?.getPurchases(
                purchaseType,
                storeEventListener
            )
        } catch (exception: Exception) {
            storeExceptionHandler.handleStoreException(
                StoreEvent.PURCHASES_REQUEST_ERROR,
                operationId,
                exception
            )
        }
    }

	fun getPurchaseHistory(operationId: Int, purchaseType: String?, maxPurchases: Int) {
        val storeEventListener = storeEventListenerFactory(operationId)
        try {
            storeBilling?.getPurchaseHistory(
                purchaseType,
                maxPurchases, storeEventListener)
        } catch (exception: Exception) {
            storeExceptionHandler.handleStoreException(
                StoreEvent.PURCHASE_HISTORY_LIST_REQUEST_ERROR,
                operationId,
                exception
            )
        }
    }

    fun getSkuDetails(operationId: Int, purchaseType: String?, skuList: List<String?>?) {
        val storeEventListener = storeEventListenerFactory(operationId)
        try {
            storeBilling?.getSkuDetails(purchaseType, skuList, storeEventListener)
        } catch (exception: Exception) {
            storeExceptionHandler.handleStoreException(
                StoreEvent.SKU_DETAILS_LIST_REQUEST_ERROR,
                operationId,
                exception
            )
        }
    }

    fun startPurchaseTracking(purchaseTypes: ArrayList<String>) {
        val storeEventListener = storeEventListenerFactory(isLifecycleListener = true)
        if (storeLifecycleListener != null) {
            stopPurchaseTracking()
        }
        storeLifecycleListener = StoreLifecycleListener(purchaseTypes, storeBilling!!, storeEventListener)
        ClientProperties.getApplication().registerActivityLifecycleCallbacks(storeLifecycleListener)
    }

	fun stopPurchaseTracking() {
        if (storeLifecycleListener != null) {
            ClientProperties.getApplication().unregisterActivityLifecycleCallbacks(
                storeLifecycleListener
            )
            storeLifecycleListener = null
        }
    }
}
