package com.unity3d.services.core.di

import android.content.Context
import android.os.Handler
import android.os.Looper
import androidx.datastore.core.DataStore
import androidx.datastore.core.DataStoreFactory
import androidx.datastore.dataStoreFile
import com.google.android.gms.net.CronetProviderInstaller
import com.unity3d.ads.adplayer.AndroidWebViewClient
import com.unity3d.ads.core.configuration.AlternativeFlowReader
import com.unity3d.ads.core.configuration.GameServerIdReader
import com.unity3d.ads.core.data.model.ByteStringSerializer
import com.unity3d.ads.core.domain.*
import com.unity3d.ads.core.domain.HandleGatewayAndroidAdResponse
import com.unity3d.ads.core.domain.events.GetOperativeEventApi
import com.unity3d.ads.core.domain.events.GetOperativeEventRequest
import com.unity3d.ads.core.domain.events.HandleGatewayAndroidEventResponse
import com.unity3d.ads.core.domain.events.HandleGatewayEventResponse
import com.unity3d.ads.core.domain.events.OperativeEventObserver
import com.unity3d.ads.gatewayclient.CommonGatewayClient
import com.unity3d.ads.gatewayclient.GatewayClient
import com.unity3d.ads.core.domain.work.BackgroundWorker
import com.unity3d.services.SDKErrorHandler
import com.unity3d.services.UnityAdsConstants
import com.unity3d.services.ads.measurements.MeasurementsService
import com.unity3d.services.ads.token.AsyncTokenStorage
import com.unity3d.services.ads.token.InMemoryAsyncTokenStorage
import com.unity3d.services.ads.token.InMemoryTokenStorage
import com.unity3d.services.ads.token.TokenStorage
import com.unity3d.services.core.device.StorageManager
import com.unity3d.services.core.device.StorageManager.StorageType
import com.unity3d.services.ads.topics.TopicsService
import com.unity3d.services.core.device.VolumeChange
import com.unity3d.services.core.device.VolumeChangeContentObserver
import com.unity3d.services.core.device.VolumeChangeMonitor
import com.unity3d.services.core.domain.ISDKDispatchers
import com.unity3d.services.core.domain.SDKDispatchers
import com.unity3d.services.core.domain.task.ConfigFileFromLocalStorage
import com.unity3d.services.core.domain.task.InitializeSDK
import com.unity3d.services.core.domain.task.InitializeStateComplete
import com.unity3d.services.core.domain.task.InitializeStateConfig
import com.unity3d.services.core.domain.task.InitializeStateConfigWithLoader
import com.unity3d.services.core.domain.task.InitializeStateCreate
import com.unity3d.services.core.domain.task.InitializeStateCreateWithRemote
import com.unity3d.services.core.domain.task.InitializeStateError
import com.unity3d.services.core.domain.task.InitializeStateLoadCache
import com.unity3d.services.core.domain.task.InitializeStateLoadWeb
import com.unity3d.services.core.domain.task.InitializeStateNetworkError
import com.unity3d.services.core.domain.task.InitializeStateReset
import com.unity3d.services.core.misc.JsonStorage
import com.unity3d.services.core.network.core.CronetClient
import com.unity3d.services.core.network.core.HttpClient
import com.unity3d.services.core.properties.ClientProperties
import com.unity3d.services.core.network.core.OkHttp3Client
import com.unity3d.services.core.request.metrics.SDKMetrics
import com.unity3d.services.core.request.metrics.SDKMetricsSender
import com.unity3d.services.core.webview.bridge.SharedInstances
import gateway.v1.NativeConfigurationOuterClass.NativeConfiguration
import gateway.v1.NativeConfigurationOuterClass.RequestRetryPolicy
import gateway.v1.NativeConfigurationOuterClass.RequestPolicy
import gateway.v1.diagnosticEventsConfiguration
import gateway.v1.nativeConfiguration
import gateway.v1.requestRetryPolicy
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.withTimeoutOrNull
import okhttp3.OkHttpClient
import ByteStringStoreOuterClass.ByteStringStore
import androidx.datastore.core.DataMigration
import androidx.datastore.core.handlers.ReplaceFileCorruptionHandler
import com.google.protobuf.ByteString
import com.unity3d.ads.adplayer.provideGetAdCacheAssetLoader
import com.unity3d.ads.adplayer.provideGetWebViewCacheAssetLoader
import com.unity3d.ads.core.data.datasource.*
import com.unity3d.ads.core.data.manager.AndroidSDKPropertiesManager
import com.unity3d.ads.core.data.manager.AndroidStorageManager
import com.unity3d.ads.core.data.manager.SDKPropertiesManager
import com.unity3d.ads.core.data.model.UniversalRequestStoreSerializer
import com.unity3d.ads.core.data.model.WebViewConfigurationStoreSerializer
import com.unity3d.ads.core.data.repository.*
import com.unity3d.ads.core.data.repository.AndroidDeviceInfoRepository
import com.unity3d.ads.core.data.repository.AndroidSessionRepository
import com.unity3d.ads.core.data.repository.DeviceInfoRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.data.repository.AndroidDiagnosticEventRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYSTEM_CRONET_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYSTEM_CRONET_SUCCESS
import com.unity3d.ads.core.domain.events.DiagnosticEventObserver
import com.unity3d.ads.core.domain.events.EventObservers
import com.unity3d.ads.core.domain.events.GetDiagnosticEventBatchRequest
import com.unity3d.ads.core.domain.events.GetDiagnosticEventRequest
import com.unity3d.ads.core.domain.events.UniversalRequestEventSender
import com.unity3d.ads.core.domain.privacy.DeveloperConsentFlattenerRulesUseCase
import com.unity3d.ads.core.domain.privacy.FlattenerRulesUseCase
import com.unity3d.ads.core.domain.privacy.LegacyUserConsentFlattenerRulesUseCase
import com.unity3d.ads.core.domain.work.DiagnosticEventRequestWorkModifier
import com.unity3d.ads.core.extensions.toByteString
import com.unity3d.services.core.preferences.AndroidPreferences
import gateway.v1.NativeConfigurationOuterClass.RequestTimeoutPolicy
import gateway.v1.requestPolicy
import gateway.v1.requestTimeoutPolicy
import com.unity3d.services.core.network.core.LegacyHttpClient
import com.unity3d.services.store.StoreWebViewEventSender
import com.unity3d.services.store.core.StoreEventListenerFactory
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.Job
import java.util.UUID
import org.chromium.net.CronetEngine
import kotlin.coroutines.resume
import kotlin.time.DurationUnit
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

/**
 * Main service provider containing and instantiating required dependencies
 */
@OptIn(ExperimentalTime::class)
object ServiceProvider : IServiceProvider {

    const val NAMED_SDK: String = "sdk"
    const val NAMED_INIT_SCOPE: String = "init_scope"
    const val NAMED_LOAD_SCOPE: String = "load_scope"
    const val NAMED_SHOW_SCOPE: String = "show_scope"
    const val NAMED_INIT_REQ: String = "init_req"
    const val NAMED_OPERATIVE_REQ: String = "op_event_req"
    const val NAMED_OTHER_REQ: String = "other_req"
    const val NAMED_AD_REQ: String = "ad_req"
    const val NAMED_PUBLIC_JOB: String = "public_job"

    const val LEGACY_PRIVACY_RULES = "legacy_privacy_rules"
    const val DEV_CONSENT_PRIVACY_RULES = "dev_consent_privacy_rules"

    // datastores
    const val DATA_STORE_GATEWAY_CACHE = "gateway_cache.pb"
    const val DATA_STORE_PRIVACY = "privacy.pb"
    const val DATA_STORE_IDFI = "idfi.pb"
    const val DATA_STORE_AUID = "auid.pb"
    const val DATA_STORE_PRIVACY_FSM = "privacy_fsm.pb"
    const val DATA_STORE_UNIVERSAL_REQUEST = "universal_request.pb"
    const val DATA_STORE_GL_INFO = "glinfo.pb"
    const val DATA_STORE_WEBVIEW_CONFIG = "webview_config.pb"
    // preferences
    const val PREF_IDFI = "unityads-installinfo"
    const val PREF_AUID = "supersonic_shared_preferen"
    const val PREF_DEFAULT = "default-migration"
    const val PREF_GL_INFO = "glinfo"
    // http
    const val GATEWAY_HOST = "gateway.unityads.unity3d.com"
    const val GATEWAY_PORT = 443
    const val CDN_CREATIVES_HOST = "cdn-creatives-cf-prd.acquire.unity3dusercontent.com"
    const val CDN_CREATIVES_PORT = 443
    const val HTTP_CACHE_DISK_SIZE = 5 * 1024 * 1024L // 5MiB
    const val HTTP_CLIENT_FETCH_TIMEOUT = 500L // 500ms

    private val serviceRegistry: IServicesRegistry = initialize()

    /**
     * Initialize the [ServicesRegistry] with the services the SDK requires
     */
    override fun initialize(): IServicesRegistry = registry {
        //context
        single<Context> { ClientProperties.getApplicationContext() }

        // logging and metrics
        single { provideSDKMetricSender() }

        // cogs
        single { provideSDKDispatchers() }
        single(NAMED_SDK) { provideSDKErrorHandler(get(), get(), get(), get()) }
        factory(NAMED_PUBLIC_JOB) { providePublicApiJob(get()) }
        factory(NAMED_INIT_SCOPE) { provideSDKScope(get(), get(NAMED_SDK), get(NAMED_PUBLIC_JOB)) }
        factory(NAMED_LOAD_SCOPE) { provideLoadScope(get(), get(NAMED_SDK), get(NAMED_PUBLIC_JOB)) }
        factory(NAMED_SHOW_SCOPE) { provideShowScope(get(), get(NAMED_SDK), get(NAMED_PUBLIC_JOB)) }

        // network
        single { provideHttpClient(get(), get(), get(), get(), get()) }

        // storage
        single(PREF_IDFI) { provideIdfiDataMigration(get()) }
        single(PREF_AUID) { provideAuidDataMigration(get()) }
        single(PREF_DEFAULT) { provideDefaultDataMigration() }
        single(PREF_GL_INFO) { provideFetchGLInfoDataMigration(get()) }
        single(DATA_STORE_GATEWAY_CACHE) {
            provideGatewayCacheDataStore(
                get(),
                get()
            )
        }
        single(DATA_STORE_PRIVACY) {
            provideUserConsentDataStore(
                get(),
                get()
            )
        }
        single(DATA_STORE_PRIVACY_FSM) {
            providePrivacyFsmDataStore(
                get(),
                get()
            )
        }
        single(DATA_STORE_IDFI) { provideIdfiDataStore(get(), get(), get(PREF_IDFI), get(PREF_DEFAULT)) }
        single(DATA_STORE_AUID) { provideAuidDataStore(get(), get(), get(PREF_AUID)) }
        single(DATA_STORE_GL_INFO) { provideGlInfoDataStore(get(), get(), get(PREF_GL_INFO)) }
        single(DATA_STORE_UNIVERSAL_REQUEST) { provideUniversalRequestDataStore(get(), get()) }
        single(DATA_STORE_WEBVIEW_CONFIG) { provideWebViewConfigurationDataStore(get(), get()) }

        // tasks
        factory { InitializeStateNetworkError(get()) }
        single { ConfigFileFromLocalStorage(get()) }
        single { InitializeStateReset(get()) }
        single { InitializeStateError(get()) }
        single { InitializeStateConfigWithLoader(get(), get(), get(), get()) }
        single { InitializeStateConfig(get(), get()) }
        single { InitializeStateCreate(get()) }
        single { InitializeStateLoadCache(get()) }
        single { InitializeStateCreateWithRemote(get()) }
        single { InitializeStateLoadWeb(get(), get(), get()) }
        single { InitializeStateComplete(get()) }
        single {
            InitializeSDK(
                get(),
                get(),
                get(),
                get(),
                get(),
                get(),
                get(),
                get(),
                get(),
                get()
            )
        }

        // repositories
        single<TokenStorage> { InMemoryTokenStorage() }
        single<AsyncTokenStorage> {
            InMemoryAsyncTokenStorage(
                null,
                Handler(Looper.getMainLooper()),
                SDKMetrics.getInstance(),
                get()
            )
        }

        // monitoring
        single<VolumeChange> { VolumeChangeContentObserver() }
        single { VolumeChangeMonitor(SharedInstances.webViewEventSender, get()) }

        // configuration
        single(StorageType.PUBLIC.name) { provideJsonStorage(StorageType.PUBLIC) }
        single(StorageType.PRIVATE.name) { provideJsonStorage(StorageType.PRIVATE) }
        single { GameServerIdReader(get(StorageType.PUBLIC.name)) }
        single { AlternativeFlowReader(get(StorageType.PUBLIC.name)) }

        // boldSDK
        single<GetRequestPolicy>(NAMED_INIT_REQ) { provideInitRequestPolicy(get()) }
        single<GetRequestPolicy>(NAMED_OPERATIVE_REQ) { provideOperativeEventRequestPolicy(get()) }
        single<GetRequestPolicy>(NAMED_OTHER_REQ) { provideOtherRequestPolicy(get()) }
        single<GetRequestPolicy>(NAMED_AD_REQ) { provideAdRequestPolicy(get()) }
        single<NativeConfiguration> { provideDefaultNativeConfiguration() }
        single { BackgroundWorker(get()) }
        single<StaticDeviceInfoDataSource> { provideStaticDeviceInfoDataSource(get(), get(DATA_STORE_IDFI), get(DATA_STORE_AUID), get(DATA_STORE_GL_INFO), get()) }
        single<AnalyticsDataSource> { AndroidAnalyticsDataSource() }
        single<LifecycleDataSource> { AndroidLifecycleDataSource() }
        single<DynamicDeviceInfoDataSource> { provideDynamicDeviceInfoDataSource(get(), get()) }
        single<PrivacyDeviceInfoDataSource> { providePrivacyDeviceInfoDataSource(get()) }
        single<MediationDataSource> { provideMediationDataSource(get(StorageType.PUBLIC.name)) }
        single<GatewayClient> { provideGatewayClient(get(), get(), get(), get()) }
        factory { AndroidWebViewClient(get(), get()) }
        single<SessionRepository> {
            provideSessionRepository(
                get(DATA_STORE_GATEWAY_CACHE),
                get(DATA_STORE_PRIVACY),
                get(DATA_STORE_PRIVACY_FSM),
                get()
            )
        }
        single(DATA_STORE_GATEWAY_CACHE) { provideByteStringDataSource(get(DATA_STORE_GATEWAY_CACHE)) }
        single(DATA_STORE_PRIVACY) { provideByteStringDataSource(get(DATA_STORE_PRIVACY)) }
        single(DATA_STORE_IDFI) { provideByteStringDataSource(get(DATA_STORE_IDFI)) }
        single(DATA_STORE_AUID) { provideByteStringDataSource(get(DATA_STORE_AUID)) }
        single(DATA_STORE_PRIVACY_FSM) { provideByteStringDataSource(get(DATA_STORE_PRIVACY_FSM)) }
        single(DATA_STORE_GL_INFO) { provideByteStringDataSource(get(DATA_STORE_GL_INFO)) }
        single<UniversalRequestDataSource> { provideUniversalRequestDataSource(get(DATA_STORE_UNIVERSAL_REQUEST)) }
        single<WebviewConfigurationDataSource> { provideWebviewConfigurationDataSource(get(DATA_STORE_WEBVIEW_CONFIG)) }
        single<DeviceInfoRepository> { provideDeviceInfoRepository(get(), get(), get()) }
        single<MediationRepository> { provideMediationRepository(get()) }
        single<HandleInvocationsFromAdViewer> { provideGetAndroidExposureUseCase(get(), get(), get(), get(), get(), get(), get(), get(), get()) }
        single<GetClientInfo> { provideGetClientInfo(get(), get()) }
        single<HandleGatewayUniversalResponse> {
            provideHandleGatewayUniversalResponse(
                get(),
                get()
            )
        }
        single<TriggerInitializeListener> { provideTriggerInitializeListener(get()) }
        single<InitializeBoldSDK> {
            provideInitializeBoldSDK(
                get(),
                get(),
                get(NAMED_INIT_REQ),
                get(),
                get(),
                get(),
                get(),
                get(),
                get(),
                get(),
                get(),
                get(),
                get()
            )
        }
        single<GetInitializationRequest> {
            provideGetInitializationRequest(
                get(),
                get(),
                get(),
                get(),
                get()
            )
        }
        single<HandleGatewayInitializationResponse> {
            provideHandleGatewayInitializationResponse(
                get(),
                get(),
                get(NAMED_INIT_SCOPE)
            )
        }
        single<GetUniversalRequestForPayLoad> { provideGetUniversalRequestForPayLoad(get()) }
        single<GetUniversalRequestSharedData> {
            provideGetUniversalRequestSharedData(
                get(),
                get(),
                get(),
                get()
            )
        }
        single<GetSharedDataTimestamps> { provideGetSharedDataTimestamps() }

        factory<Load> { provideLoad(get(), get(), get(), get(NAMED_AD_REQ), get(), get(), get(), get()) }
        factory<LoadAdMarkup> { provideLoadAdMarkup(get(), get(), get(NAMED_AD_REQ), get(), get(), get(), get(), get()) }
        single<Refresh> { provideRefresh(get(), get(), get(NAMED_AD_REQ), get()) }
        factory<LegacyLoadUseCase> { provideLegacyLoadUseCase(get(), get(), get(), get(), get(), get()) }
        single<GetAdPlayerConfigRequest> { provideGetAdPlayerConfigRequest(get()) }
        single<GetAdRequest> { provideGetAdRequest(get(), get(), get(), get(), get()) }
        single<GetAdDataRefreshRequest> { provideGetAdDataRefreshRequest(get(), get(), get(), get()) }
        factory<HandleGatewayAdResponse> { provideHandleGatewayAdResponse(get(), get(), get(), get(), get(), get(), get(), get(), get(), get(), get(), get()) }
        single<AdRepository> { provideAdRepository() }
        single<CampaignRepository> { provideCampaignStateRepository(get()) }

        single<GetOperativeEventApi> { GetOperativeEventApi(get(), get()) }
        single<GetOperativeEventRequest> { GetOperativeEventRequest(get(), get(), get(), get()) }
        single<HandleGatewayEventResponse> { HandleGatewayAndroidEventResponse() }
        single<OperativeEventRepository> { OperativeEventRepository() }
        single<OperativeEventObserver> { provideOperativeEventObserver(get(), get(), get(), get(), get()) }
        single<GetDiagnosticEventRequest> { GetDiagnosticEventRequest(get()) }
        single<DiagnosticEventRepository> { AndroidDiagnosticEventRepository() }
        single<SendDiagnosticEvent> { AndroidSendDiagnosticEvent(get(), get()) }
        factory<DiagnosticEventRequestWorkModifier> { DiagnosticEventRequestWorkModifier(get(), get()) }
        single<GetDiagnosticEventBatchRequest> { GetDiagnosticEventBatchRequest() }
        single<DiagnosticEventObserver> { provideDiagnosticEventObserver(get(), get(), get(), get(), get(), get()) }
        single<EventObservers> { EventObservers(get(), get()) }
        single<UniversalRequestEventSender> { UniversalRequestEventSender(get(), get()) }

        single<Show> { provideShow(get(), get(), get()) }
        single<LegacyShowUseCase> { provideLegacyShowUseCase(get(), get(), get(), get(), get(), get(), get()) }

        factory<AndroidGetWebViewContainerUseCase> {
            provideGetWebViewContainerUseCase(
                get(),
                get(),
                get(),
                get(),
            )
        }
        single<SendWebViewClientErrorDiagnostics> { provideSendWebViewClientErrorDiagnostics(get()) }
        single<GetWebViewBridgeUseCase> { provideGetWebViewBridgeUseCase(get()) }
        single<GetAndroidAdPlayerContext> { GetAndroidAdPlayerContext(get(), get()) }
        single<GetInitializationCompletedRequest> {
            provideGetInitializationCompletedRequest(
                get(),
                get()
            )
        }
        single<TriggerInitializationCompletedRequest> {
            provideTriggerInitializationCompletedRequest(
                get(),
                get(NAMED_INIT_REQ),
                get(),
                get()
            )
        }
        single<HandleOpenUrl> { AndroidHandleOpenUrl(get()) }
        single<GetOpenGLRendererInfo> { provideGetOpenGLRendererInfo(get()) }
        single<ExecuteAdViewerRequest> { provideExecuteAdViewerRequest(get(), get()) }
        single<GetPrivacyUpdateRequest> { GetPrivacyUpdateRequest(get()) }
        single<SendPrivacyUpdateRequest> { SendPrivacyUpdateRequest(get(), get(NAMED_OTHER_REQ), get()) }

        single<FlattenerRulesUseCase>(LEGACY_PRIVACY_RULES) { LegacyUserConsentFlattenerRulesUseCase() }
        single<LegacyUserConsentDataSource> { AndroidLegacyUserConsentDataSource(get(LEGACY_PRIVACY_RULES), get(StorageType.PRIVATE.name)) }
        single<LegacyUserConsentRepository> { AndroidLegacyUserConsentRepository(get()) }

        single<FlattenerRulesUseCase>(DEV_CONSENT_PRIVACY_RULES) { DeveloperConsentFlattenerRulesUseCase() }
        single<DeveloperConsentDataSource> { AndroidDeveloperConsentDataSource(get(DEV_CONSENT_PRIVACY_RULES), get(StorageType.PUBLIC.name)) }
        single<DeveloperConsentRepository> { AndroidDeveloperConsentRepository(get()) }
        single<com.unity3d.ads.core.data.manager.StorageManager> { AndroidStorageManager() }
        single<SDKPropertiesManager> { AndroidSDKPropertiesManager() }
        single { GetLatestWebViewConfiguration(get()) }
        single { provideGetWebViewCacheAssetLoader(get()) }
        single { provideGetAdCacheAssetLoader(get()) }
        single<GetHeaderBiddingToken> { CommonGetHeaderBiddingToken(get(), get() , get(), get(), get(), get()) }
        single<GetByteStringId> { AndroidGenerateByteStringId() }
        single<GetInitializationState> { provideGetInitializationState(get()) }

        // android privacy sandbox
        single { MeasurementsService(ClientProperties.getApplicationContext(), get(), SharedInstances.webViewEventSender) }
        single { TopicsService(ClientProperties.getApplicationContext(), get(), SharedInstances.webViewEventSender) }

        // store transactions
        single { StoreWebViewEventSender() }
        single { StoreEventListenerFactory(get(), get()) }
    }

    private fun provideMediationDataSource(publicStorage: JsonStorage): MediationDataSource {
        return AndroidMediationDataSource(publicStorage)
    }

    private fun provideMediationRepository(mediationDataSource: MediationDataSource): MediationRepository {
        return AndroidMediationRepository(mediationDataSource)
    }

    private fun provideInitRequestPolicy(sessionRepository: SessionRepository): GetRequestPolicy {
        return GetInitRequestPolicy(sessionRepository)
    }

    private fun provideOtherRequestPolicy(sessionRepository: SessionRepository): GetRequestPolicy {
        return GetOtherRequestPolicy(sessionRepository)
    }

    private fun provideAdRequestPolicy(sessionRepository: SessionRepository): GetRequestPolicy {
        return GetAdRequestPolicy(sessionRepository)
    }

    private fun provideOperativeEventRequestPolicy(sessionRepository: SessionRepository): GetRequestPolicy {
        return GetOperativeEventRequestPolicy(sessionRepository)
    }

    private fun provideGetWebViewBridgeUseCase(dispatchers: ISDKDispatchers): GetWebViewBridgeUseCase {
        return GetCommonWebViewBridgeUseCase(dispatchers.default)

    }

    private fun provideOperativeEventObserver(getUniversalRequestForPayLoad: GetUniversalRequestForPayLoad, dispatchers: ISDKDispatchers, operativeEventRepository: OperativeEventRepository, universalRequestDataSource: UniversalRequestDataSource, backgroundWorker: BackgroundWorker): OperativeEventObserver {
        return OperativeEventObserver(
            getUniversalRequestForPayLoad,
            dispatchers.default,
            operativeEventRepository,
            universalRequestDataSource,
            backgroundWorker
        )
    }

    private fun provideDiagnosticEventObserver(getUniversalRequestForPayLoad: GetUniversalRequestForPayLoad, getDiagnosticEventBatchRequest: GetDiagnosticEventBatchRequest, dispatchers: ISDKDispatchers, diagnosticEventRepository: DiagnosticEventRepository, universalRequestDataSource: UniversalRequestDataSource, backgroundWorker: BackgroundWorker): DiagnosticEventObserver {
        return DiagnosticEventObserver(
            getUniversalRequestForPayLoad,
            getDiagnosticEventBatchRequest,
            dispatchers.default,
            diagnosticEventRepository,
            universalRequestDataSource,
            backgroundWorker
        )
    }

    private fun provideTriggerInitializeListener(dispatchers: ISDKDispatchers): TriggerInitializeListener {
        return TriggerInitializeListener(dispatchers.main)
    }

    private fun provideGetWebViewContainerUseCase(
        context: Context,
        androidWebViewClient: AndroidWebViewClient,
        sendWebViewClientErrorDiagnostics: SendWebViewClientErrorDiagnostics,
        dispatchers: ISDKDispatchers,
    ): AndroidGetWebViewContainerUseCase {
        return AndroidGetWebViewContainerUseCase(
            context,
            androidWebViewClient,
            sendWebViewClientErrorDiagnostics,
            dispatchers.main,
            dispatchers.default,
        )
    }

    private fun provideSendWebViewClientErrorDiagnostics(
        sendDiagnosticEvent: SendDiagnosticEvent
    ) : SendWebViewClientErrorDiagnostics {
        return AndroidSendWebViewClientErrorDiagnostics(sendDiagnosticEvent)
    }

    /**
     * @return the current [IServicesRegistry] tied to the [ServiceProvider]
     */
    override fun getRegistry(): IServicesRegistry = serviceRegistry

    /**
     * Provides an SDK wide thread handling for tasks to be executed on
     */
    private fun provideSDKDispatchers(): ISDKDispatchers {
        return SDKDispatchers()
    }

    /**
     * SDK [CoroutineExceptionHandler] for capturing any unhandled exceptions at a parent level
     */
    private fun provideSDKErrorHandler(
        dispatchers: ISDKDispatchers,
        alternativeFlowReader: AlternativeFlowReader,
        sendDiagnosticEvent: SendDiagnosticEvent,
        sdkMetricsSender: SDKMetricsSender
    ): CoroutineExceptionHandler {
        return SDKErrorHandler(dispatchers.io, alternativeFlowReader, sendDiagnosticEvent, sdkMetricsSender)
    }

    /**
     * SDK parent [CoroutineScope], won't crash on exceptions, will catch/handle all children's exceptions
     */
    private fun provideSDKScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        job: Job,
    ): CoroutineScope {
        return CoroutineScope(dispatchers.default + job + CoroutineName(NAMED_INIT_SCOPE) + errorHandler)
    }

    private fun provideLoadScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        job: Job,
    ): CoroutineScope {
        return CoroutineScope(dispatchers.default + job + CoroutineName(NAMED_LOAD_SCOPE) +errorHandler)
    }

    private fun provideShowScope(
        dispatchers: ISDKDispatchers,
        errorHandler: CoroutineExceptionHandler,
        job: Job,
    ): CoroutineScope {
        return CoroutineScope(dispatchers.default + job + CoroutineName(NAMED_SHOW_SCOPE) + errorHandler)
    }

    private fun providePublicApiJob(
        diagnosticEventRepository: DiagnosticEventRepository,
    ): Job = Job().apply {
            invokeOnCompletion {
                diagnosticEventRepository.flush()
            }
        }

    private fun provideSDKMetricSender(): SDKMetricsSender {
        // TODO: Switch implementation to use injection for HttpClient, Configuration, etc.
        return SDKMetrics.getInstance()
    }

    /**
     * Provides an [HttpClient] to be used for the network needs of the SDK
     */
    private fun provideHttpClient(
        configFileFromLocalStorage: ConfigFileFromLocalStorage,
        alternativeFlowReader: AlternativeFlowReader,
        dispatchers: ISDKDispatchers,
        sendDiagnosticEvent: SendDiagnosticEvent,
        context: Context
    ): HttpClient = runBlocking {
        val isAlternativeFlowEnabled = alternativeFlowReader()
        if (isAlternativeFlowEnabled) {
            val startTime = TimeSource.Monotonic.markNow()
            val client = withTimeoutOrNull(HTTP_CLIENT_FETCH_TIMEOUT) {
                buildNetworkClient(context, dispatchers)
            }
            val diagnosticResult = if (client == null) SYSTEM_CRONET_FAILURE else SYSTEM_CRONET_SUCCESS
            sendDiagnosticEvent(diagnosticResult, startTime.elapsedNow().toDouble(DurationUnit.MILLISECONDS))
            client ?: OkHttp3Client(dispatchers, OkHttpClient())
        } else {
            val config = runBlocking {
                runCatching { configFileFromLocalStorage(ConfigFileFromLocalStorage.Params()) }.getOrNull()?.getOrNull()
            }
            if (config?.experiments?.isOkHttpEnabled == true) {
                OkHttp3Client(dispatchers, OkHttpClient())
            } else {
                LegacyHttpClient(dispatchers)
            }
        }
    }

    private suspend fun buildNetworkClient(
        context: Context,
        dispatchers: ISDKDispatchers
    ): HttpClient = suspendCancellableCoroutine { continuation ->
        CronetProviderInstaller.installProvider(context).addOnCompleteListener {
            if (it.isSuccessful) {
                val cronetEngine = CronetEngine.Builder(context)
                    .setStoragePath(context.filesDir.absolutePath)
                    .enableHttpCache(CronetEngine.Builder.HTTP_CACHE_DISK, HTTP_CACHE_DISK_SIZE) // HTTP_CACHE_DISK provides 0-RTT support
                    .enableQuic(true)
                    .addQuicHint(GATEWAY_HOST, GATEWAY_PORT, GATEWAY_PORT)
                    .addQuicHint(CDN_CREATIVES_HOST, CDN_CREATIVES_PORT, CDN_CREATIVES_PORT)
                    .build()
                continuation.resume(CronetClient(cronetEngine, dispatchers))
            } else {
                continuation.resume(OkHttp3Client(dispatchers, OkHttpClient()))
            }
        }
    }

    private fun provideJsonStorage(storageType: StorageType): JsonStorage {
        check(StorageManager.init(ClientProperties.getApplicationContext())) {
            "StorageManager failed to initialize"
        }
        return StorageManager.getStorage(storageType)
    }

    private fun provideInitializeBoldSDK(
        dispatchers: ISDKDispatchers,
        getInitializeRequest: GetInitializationRequest,
        getInitRequestPolicy: GetRequestPolicy,
        handleGatewayInitializationResponse: HandleGatewayInitializationResponse,
        gatewayClient: GatewayClient,
        sessionRepository: SessionRepository,
        eventObservers: EventObservers,
        triggerInitializeListener: TriggerInitializeListener,
        sendDiagnosticEvent: SendDiagnosticEvent,
        diagnosticEventRepository: DiagnosticEventRepository,
        deviceInfoRepository: DeviceInfoRepository,
        storageManager: com.unity3d.ads.core.data.manager.StorageManager,
        sdkPropertiesManager: SDKPropertiesManager
    ): InitializeBoldSDK {
        return InitializeAndroidBoldSDK(
            dispatchers.default,
            getInitializeRequest,
            getInitRequestPolicy,
            handleGatewayInitializationResponse,
            gatewayClient,
            sessionRepository,
            eventObservers,
            triggerInitializeListener,
            sendDiagnosticEvent,
            diagnosticEventRepository,
            deviceInfoRepository,
            storageManager,
            sdkPropertiesManager
        )
    }

    private fun provideGetInitializationRequest(
        getUniversalRequestForPayLoad: GetUniversalRequestForPayLoad,
        getClientInfo: GetClientInfo,
        sessionRepository: SessionRepository,
        deviceInfoRepository: DeviceInfoRepository,
        legacyUserConsentRepository: LegacyUserConsentRepository
    ): GetInitializationRequest {
        return GetAndroidInitializationRequest(
            getUniversalRequestForPayLoad,
            getClientInfo,
            sessionRepository,
            deviceInfoRepository,
            legacyUserConsentRepository
        )
    }

    private fun provideHandleGatewayUniversalResponse(
        sessionRepository: SessionRepository,
        deviceInfoRepository: DeviceInfoRepository
    ): HandleGatewayUniversalResponse {
        return HandleAndroidGatewayUniversalResponse(sessionRepository, deviceInfoRepository)
    }

    private fun provideHandleGatewayInitializationResponse(
        triggerInitializationCompletedRequest: TriggerInitializationCompletedRequest,
        sessionRepository: SessionRepository,
        sdkScope: CoroutineScope
    ): HandleGatewayInitializationResponse {
        return HandleAndroidGatewayInitializationResponse(
            triggerInitializationCompletedRequest,
            sessionRepository,
            sdkScope
        )
    }

    private fun provideUniversalRequestDataSource(dataStore: DataStore<UniversalRequestStoreOuterClass.UniversalRequestStore>): UniversalRequestDataSource {
        return UniversalRequestDataSource(dataStore)
    }

    private fun provideWebviewConfigurationDataSource(dataStore: DataStore<WebviewConfigurationStore.WebViewConfigurationStore>): WebviewConfigurationDataSource {
        return WebviewConfigurationDataSource(dataStore)
    }

    private fun provideByteStringDataSource(dataStore: DataStore<ByteStringStore>): ByteStringDataSource {
        return AndroidByteStringDataSource(dataStore)
    }

    private fun provideSessionRepository(
        gatewayCacheDataSource: ByteStringDataSource,
        privacyDataSource: ByteStringDataSource,
        privacyFsmDataSource: ByteStringDataSource,
        nativeConfiguration: NativeConfiguration
    ): SessionRepository {
        return AndroidSessionRepository(
            gatewayCacheDataSource,
            privacyDataSource,
            privacyFsmDataSource,
            nativeConfiguration
        )
    }

    private fun provideGetClientInfo(sessionRepository: SessionRepository, mediationRepository: MediationRepository): GetClientInfo {
        return GetAndroidClientInfo(sessionRepository, mediationRepository)
    }

    private fun provideGatewayClient(
        httpClient: HttpClient,
        handleGatewayUniversalResponse: HandleGatewayUniversalResponse,
        sendDiagnosticEvent: SendDiagnosticEvent,
        sessionRepository: SessionRepository,
    ): GatewayClient {
        return CommonGatewayClient(
            httpClient,
            handleGatewayUniversalResponse,
            sendDiagnosticEvent,
            sessionRepository,
        )
    }

    private fun provideGetUniversalRequestForPayLoad(getUniversalRequestSharedData: GetUniversalRequestSharedData): GetUniversalRequestForPayLoad {
        return GetAndroidUniversalRequestForPayLoad(getUniversalRequestSharedData)
    }

    private fun provideGetUniversalRequestSharedData(
        getSharedDataTimestamps: GetSharedDataTimestamps,
        sessionRepository: SessionRepository,
        deviceInfoRepository: DeviceInfoRepository,
        developerConsentRepository: DeveloperConsentRepository
    ): GetUniversalRequestSharedData {
        return GetAndroidUniversalRequestSharedData(
            getSharedDataTimestamps,
            sessionRepository,
            deviceInfoRepository,
            developerConsentRepository
        )
    }

    private fun provideGetSharedDataTimestamps(): GetSharedDataTimestamps {
        return GetAndroidSharedDataTimestamps()
    }

    private fun provideDeviceInfoRepository(
        staticDeviceInfoDataSource: StaticDeviceInfoDataSource,
        dynamicDeviceInfoDataSource: DynamicDeviceInfoDataSource,
        privacyDeviceInfoDataSource: PrivacyDeviceInfoDataSource
    ): DeviceInfoRepository {
        return AndroidDeviceInfoRepository(
            staticDeviceInfoDataSource,
            dynamicDeviceInfoDataSource,
            privacyDeviceInfoDataSource
        )
    }

    private fun provideStaticDeviceInfoDataSource(
        context: Context,
        idfiStore: ByteStringDataSource,
        auidStore: ByteStringDataSource,
        glInfoStore: ByteStringDataSource,
        analyticsDataSource: AnalyticsDataSource
    ): StaticDeviceInfoDataSource {
        return AndroidStaticDeviceInfoDataSource(
            context,
            idfiStore,
            auidStore,
            glInfoStore,
            analyticsDataSource
        )
    }

    private fun provideDynamicDeviceInfoDataSource(
        context: Context,
        lifecycleDataSource: LifecycleDataSource
    ): DynamicDeviceInfoDataSource {
        return AndroidDynamicDeviceInfoDataSource(context, lifecycleDataSource)
    }

    private fun providePrivacyDeviceInfoDataSource(
        context: Context
    ): PrivacyDeviceInfoDataSource {
        return AndroidPrivacyDeviceInfoDataSource(context)
    }

    private fun provideLoad(
        dispatchers: ISDKDispatchers,
        getAdRequest: GetAdRequest,
        getAdPlayerConfigRequest: GetAdPlayerConfigRequest,
        getOtherRequestPolicy: GetRequestPolicy,
        handleGatewayAdResponse: HandleGatewayAdResponse,
        sessionRepository: SessionRepository,
        gatewayClient: GatewayClient,
        adRepository: AdRepository
    ): Load {
        return AndroidLoad(
            dispatchers.default,
            getAdRequest,
            getAdPlayerConfigRequest,
            getOtherRequestPolicy,
            handleGatewayAdResponse,
            sessionRepository,
            gatewayClient,
            adRepository,
        )
    }

    private fun provideLoadAdMarkup(
        dispatchers: ISDKDispatchers,
        getAdRequest: GetAdRequest,
        getOtherRequestPolicy: GetRequestPolicy,
        handleGatewayAdResponse: HandleGatewayAdResponse,
        sessionRepository: SessionRepository,
        gatewayClient: GatewayClient,
        adRepository: AdRepository,
        httpClient: HttpClient
    ): LoadAdMarkup {
        return LoadAdMarkup(
            dispatchers.default,
            getAdRequest,
            getOtherRequestPolicy,
            handleGatewayAdResponse,
            sessionRepository,
            gatewayClient,
            adRepository,
            httpClient
        )
    }

    private fun provideRefresh(
        dispatchers: ISDKDispatchers,
        getAdDataRefreshRequest: GetAdDataRefreshRequest,
        getRequestPolicy: GetRequestPolicy,
        gatewayClient: GatewayClient,
    ): Refresh {
        return AndroidRefresh(
            dispatchers.default,
            getAdDataRefreshRequest,
            getRequestPolicy,
            gatewayClient,
        )
    }

    private fun provideGetAdRequest(
        getUniversalRequestForPayLoad: GetUniversalRequestForPayLoad,
        sessionRepository: SessionRepository,
        deviceInfoRepository: DeviceInfoRepository,
        campaignRepository: CampaignRepository,
        webviewConfigurationDataSource: WebviewConfigurationDataSource
    ): GetAdRequest {
        return GetAndroidAdRequest(
            getUniversalRequestForPayLoad,
            sessionRepository,
            deviceInfoRepository,
            campaignRepository,
            webviewConfigurationDataSource
        )
    }

    private fun provideGetAdPlayerConfigRequest(
        getUniversalRequestForPayLoad: GetUniversalRequestForPayLoad,
    ): GetAdPlayerConfigRequest {
        return GetAndroidAdPlayerConfigRequest(getUniversalRequestForPayLoad)
    }

    private fun provideGetAdDataRefreshRequest(
        getUniversalRequestForPayLoad: GetUniversalRequestForPayLoad,
        sessionRepository: SessionRepository,
        deviceInfoRepository: DeviceInfoRepository,
        campaignRepository: CampaignRepository,
    ): GetAdDataRefreshRequest {
        return GetAndroidAdDataRefreshRequest(
            getUniversalRequestForPayLoad,
            sessionRepository,
            deviceInfoRepository,
            campaignRepository,
        )
    }

    private fun provideHandleGatewayAdResponse(
        adRepository: AdRepository,
        getWebViewContainerUseCase: AndroidGetWebViewContainerUseCase,
        getWebViewBridge: GetWebViewBridgeUseCase,
        dispatchers: ISDKDispatchers,
        deviceInfoRepository: DeviceInfoRepository,
        getHandleInvocationsFromAdViewer: HandleInvocationsFromAdViewer,
        sessionRepository: SessionRepository,
        campaignRepository: CampaignRepository,
        executeAdViewerRequest: ExecuteAdViewerRequest,
        sendDiagnosticEvent: SendDiagnosticEvent,
        operativeEventApi: GetOperativeEventApi,
        latestWebViewConfigurationUseCase: GetLatestWebViewConfiguration,
    ): HandleGatewayAdResponse {
        return HandleGatewayAndroidAdResponse(
            adRepository,
            getWebViewContainerUseCase,
            getWebViewBridge,
            dispatchers.default,
            deviceInfoRepository,
            getHandleInvocationsFromAdViewer,
            sessionRepository,
            campaignRepository,
            executeAdViewerRequest,
            sendDiagnosticEvent,
            operativeEventApi,
            latestWebViewConfigurationUseCase,
        )
    }

    private fun provideAdRepository(): AdRepository {
        return AndroidAdRepository()
    }

    private fun provideCampaignStateRepository(
        getSharedDataTimestamps: GetSharedDataTimestamps
    ): CampaignRepository {
        return AndroidCampaignRepository(getSharedDataTimestamps)
    }

    private fun provideShow(
        adRepository: AdRepository,
        gameServerIdReader: GameServerIdReader,
        sendDiagnosticEvent: SendDiagnosticEvent
    ): Show {
        return AndroidShow(adRepository, gameServerIdReader, sendDiagnosticEvent)
    }

    private fun provideGetAndroidExposureUseCase(
        getAndroidAdPlayerContext: GetAndroidAdPlayerContext,
        getOperativeEventApi: GetOperativeEventApi,
        refresh: Refresh,
        handleOpenUrl: HandleOpenUrl,
        sessionRepository: SessionRepository,
        deviceInfoRepository: DeviceInfoRepository,
        campaignRepository: CampaignRepository,
        sendPrivacyUpdateRequest: SendPrivacyUpdateRequest,
        sendDiagnosticEvent: SendDiagnosticEvent,
    ): HandleInvocationsFromAdViewer {
        return HandleInvocationsFromAdViewer(getAndroidAdPlayerContext, getOperativeEventApi, refresh, handleOpenUrl, sessionRepository, deviceInfoRepository, campaignRepository, sendPrivacyUpdateRequest, sendDiagnosticEvent)
    }

    private fun provideGatewayCacheDataStore(
        context: Context,
        dispatchers: ISDKDispatchers
    ): DataStore<ByteStringStore> {
        return provideByteStringDataStore(context, dispatchers, DATA_STORE_GATEWAY_CACHE)
    }

    private fun provideUserConsentDataStore(
        context: Context,
        dispatchers: ISDKDispatchers
    ): DataStore<ByteStringStore> {
        return provideByteStringDataStore(context, dispatchers, DATA_STORE_PRIVACY)
    }

    private fun providePrivacyFsmDataStore(
        context: Context,
        dispatchers: ISDKDispatchers
    ): DataStore<ByteStringStore> {
        return provideByteStringDataStore(context, dispatchers, DATA_STORE_PRIVACY_FSM)
    }

    private fun provideIdfiDataStore(
        context: Context,
        dispatchers: ISDKDispatchers,
        idfiMigration: DataMigration<ByteStringStore>,
        defaultIdfi: DataMigration<ByteStringStore>
    ): DataStore<ByteStringStore> {
        return DataStoreFactory.create(
            serializer = ByteStringSerializer(),
            produceFile = { context.dataStoreFile(DATA_STORE_IDFI) },
            migrations = listOf(idfiMigration, defaultIdfi),
            corruptionHandler = ReplaceFileCorruptionHandler(
                produceNewData = {
                    val idfi = UUID.randomUUID()
                    AndroidPreferences.setString("unityads-installinfo", "unityads-idfi", idfi.toString()) // legacy support
                    ByteStringStore.newBuilder().setData(idfi.toByteString()).build()
                }
            ),
            scope = CoroutineScope(dispatchers.io + SupervisorJob())
        )
    }

    private fun provideAuidDataStore(
        context: Context,
        dispatchers: ISDKDispatchers,
        auidMigration: DataMigration<ByteStringStore>
    ): DataStore<ByteStringStore> {
        return DataStoreFactory.create(
            serializer = ByteStringSerializer(),
            produceFile = { context.dataStoreFile(DATA_STORE_AUID) },
            migrations = listOf(auidMigration),
            corruptionHandler = ReplaceFileCorruptionHandler(
                produceNewData = {
                    ByteStringStore.newBuilder().setData(ByteString.empty()).build()
                }
            ),
            scope = CoroutineScope(dispatchers.io + SupervisorJob())
        )
    }


    private fun provideGlInfoDataStore(
        context: Context,
        dispatchers: ISDKDispatchers,
        fetchGLInfo: DataMigration<ByteStringStore>
    ): DataStore<ByteStringStore> {
        return DataStoreFactory.create(
            serializer = ByteStringSerializer(),
            produceFile = { context.dataStoreFile(DATA_STORE_GL_INFO) },
            migrations = listOf(fetchGLInfo),
            scope = CoroutineScope(dispatchers.io + SupervisorJob())
        )
    }

    private fun provideAuidDataMigration(context: Context): DataMigration<ByteStringStore> {
        return ForcefulPreservingByteStringPreferenceMigration(
            context,
            AndroidStaticDeviceInfoDataSource.PREF_KEY_SUPERSONIC,
            AndroidStaticDeviceInfoDataSource.PREF_KEY_AUID,
            GetAuidData()
        )
    }


    private fun provideIdfiDataMigration(context: Context): DataMigration<ByteStringStore> {
        return PreservingByteStringPreferenceMigration(
            context,
            AndroidStaticDeviceInfoDataSource.PREF_KEY_INSTALLINFO,
            AndroidStaticDeviceInfoDataSource.PREF_KEY_IDFI,
            GetIdfiData()
        )
    }

    private fun provideDefaultDataMigration(): DataMigration<ByteStringStore> {
        return DefaultByteStringMigration(
            AndroidStaticDeviceInfoDataSource.PREF_KEY_INSTALLINFO,
            AndroidStaticDeviceInfoDataSource.PREF_KEY_IDFI,
            GetIdfiData()
        )
    }

    private fun provideFetchGLInfoDataMigration(getOpenGLRendererInfo: GetOpenGLRendererInfo): DataMigration<ByteStringStore> {
        return FetchGLInfoDataMigration(getOpenGLRendererInfo)
    }

    private fun provideGetOpenGLRendererInfo(sessionRepository: SessionRepository): GetOpenGLRendererInfo {
        return GetAndroidOpenGLRendererInfo(sessionRepository)
    }

    private fun provideByteStringDataStore(
        context: Context,
        dispatchers: ISDKDispatchers,
        dataStoreFile: String
    ): DataStore<ByteStringStore> {
        return DataStoreFactory.create(
            serializer = ByteStringSerializer(),
            produceFile = { context.dataStoreFile(dataStoreFile) },
            corruptionHandler = null,
            scope = CoroutineScope(dispatchers.io + SupervisorJob())
        )
    }

    private fun provideUniversalRequestDataStore(
        context: Context,
        dispatchers: ISDKDispatchers
    ): DataStore<UniversalRequestStoreOuterClass.UniversalRequestStore> {
        return DataStoreFactory.create(
            serializer = UniversalRequestStoreSerializer(),
            produceFile = { context.dataStoreFile(DATA_STORE_UNIVERSAL_REQUEST) },
            corruptionHandler = null,
            scope = CoroutineScope(dispatchers.io + SupervisorJob())
        )
    }

    private fun provideWebViewConfigurationDataStore(
        context: Context,
        dispatchers: ISDKDispatchers
    ): DataStore<WebviewConfigurationStore.WebViewConfigurationStore> {
        return DataStoreFactory.create(
            serializer = WebViewConfigurationStoreSerializer(),
            produceFile = { context.dataStoreFile(DATA_STORE_WEBVIEW_CONFIG) },
            corruptionHandler = null,
            scope = CoroutineScope(dispatchers.io + SupervisorJob())
        )
    }

    private fun provideGetInitializationCompletedRequest(
        getUniversalRequestForPayLoad: GetUniversalRequestForPayLoad,
        deviceInfoRepository: DeviceInfoRepository
    ): GetInitializationCompletedRequest {
        return GetAndroidInitializationCompletedRequest(
            getUniversalRequestForPayLoad,
            deviceInfoRepository
        )
    }

    private fun provideTriggerInitializationCompletedRequest(
        getInitializationCompletedRequest: GetInitializationCompletedRequest,
        getInitRequestPolicy: GetRequestPolicy,
        gatewayClient: GatewayClient,
        sendDiagnosticEvent: SendDiagnosticEvent
    ): TriggerInitializationCompletedRequest {
        return TriggerAndroidInitializationCompletedRequest(
            getInitializationCompletedRequest,
            getInitRequestPolicy,
            gatewayClient,
            sendDiagnosticEvent
        )
    }

    private fun getDefaultRequestPolicy(): RequestPolicy {
        return requestPolicy {
            retryPolicy = getDefaultRequestRetryPolicy()
            timeoutPolicy = getDefaultRequestTimeoutPolicy()
        }
    }

    private fun getDefaultRequestRetryPolicy(): RequestRetryPolicy {
        return requestRetryPolicy {
            maxDuration = UnityAdsConstants.RequestPolicy.RETRY_MAX_DURATION
            retryWaitBase = UnityAdsConstants.RequestPolicy.RETRY_WAIT_BASE
            retryJitterPct = UnityAdsConstants.RequestPolicy.RETRY_JITTER_PCT
            shouldStoreLocally = UnityAdsConstants.RequestPolicy.SHOULD_STORE_LOCALLY
        }
    }

    private fun getDefaultRequestTimeoutPolicy(): RequestTimeoutPolicy {
        return requestTimeoutPolicy {
            connectTimeoutMs = UnityAdsConstants.RequestPolicy.CONNECT_TIMEOUT_MS
            readTimeoutMs = UnityAdsConstants.RequestPolicy.READ_TIMEOUT_MS
            writeTimeoutMs = UnityAdsConstants.RequestPolicy.WRITE_TIMEOUT_MS
        }
    }

    private fun provideDefaultNativeConfiguration(): NativeConfiguration {
        return nativeConfiguration {
            initPolicy = getDefaultRequestPolicy()
            adPolicy = getDefaultRequestPolicy()
            otherPolicy = getDefaultRequestPolicy()
            operativeEventPolicy = getDefaultRequestPolicy()
            diagnosticEvents = diagnosticEventsConfiguration {
                enabled = true
                maxBatchSize = 1
                maxBatchIntervalMs = 1000
                ttmEnabled = false
            }
        }
    }

    private fun provideExecuteAdViewerRequest(
        dispatchers: ISDKDispatchers,
        httpClient: HttpClient
    ) : ExecuteAdViewerRequest {
        return AndroidExecuteAdViewerRequest(dispatchers.io, httpClient)
    }

    private fun provideLegacyShowUseCase(
        dispatcher: ISDKDispatchers,
        show: Show,
        adRepository: AdRepository,
        sendDiagnosticEvent: SendDiagnosticEvent,
        operativeEventApi: GetOperativeEventApi,
        getInitializationState: GetInitializationState,
        sessionRepository: SessionRepository
    ) : LegacyShowUseCase {
        return LegacyShowUseCase(dispatcher.main, show, adRepository, sendDiagnosticEvent, operativeEventApi, getInitializationState, sessionRepository)
    }

    private fun provideLegacyLoadUseCase(
        dispatcher: ISDKDispatchers,
        load: Load,
        adRepository: AdRepository,
        getInitializationState: GetInitializationState,
        sendDiagnosticEvent: SendDiagnosticEvent,
        sessionRepository: SessionRepository
    ) : LegacyLoadUseCase {
        return LegacyLoadUseCase(dispatcher.main, load, sendDiagnosticEvent, getInitializationState, sessionRepository, adRepository)
    }

    private fun provideGetInitializationState(
        sessionRepository: SessionRepository
    ): GetInitializationState {
        return CommonGetInitializationState(sessionRepository)
    }

}
