package com.unity3d.ads.core.domain

import android.content.Context
import com.unity3d.ads.IUnityAdsShowListener
import com.unity3d.ads.UnityAds
import com.unity3d.ads.UnityAdsShowOptions
import com.unity3d.ads.adplayer.EmbeddableAdPlayer
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.data.model.ShowEvent
import com.unity3d.ads.adplayer.model.ShowStatus
import com.unity3d.ads.core.data.model.InitializationState
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_NOT_INITIALIZED
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_PLACEMENT_NULL
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.OPERATION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_AD_OBJECT_NOT_FOUND
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_AD_PLAYER_SCOPE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_ALREADY_SHOWING
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_CODE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_NOT_INITIALIZED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_OPPORTUNITY_ID
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_PLACEMENT_NULL
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_PLACEMENT_VALIDATION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_TIMEOUT
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_UNCAUGHT_EXCEPTION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SHOW_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SHOW_HAS_STARTED
import com.unity3d.ads.core.domain.events.GetOperativeEventApi
import com.unity3d.ads.core.extensions.*
import com.unity3d.services.core.log.DeviceLog
import gateway.v1.OperativeEventRequestOuterClass.OperativeEventErrorType
import gateway.v1.OperativeEventRequestOuterClass.OperativeEventType
import gateway.v1.operativeEventErrorData
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.isActive
import kotlinx.coroutines.withContext
import java.util.UUID
import kotlin.time.ExperimentalTime
import kotlin.time.TimeMark
import kotlin.time.TimeSource

@OptIn(ExperimentalTime::class)
internal class LegacyShowUseCase(
    private val dispatcher: CoroutineDispatcher,
    private val show: Show,
    private val adRepository: AdRepository,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val getOperativeEventApi: GetOperativeEventApi,
    private val getInitializationState: GetInitializationState,
    private val sessionRepository: SessionRepository
) {
    @Volatile
    private var isFullscreenAdShowing = false
    private val hasStarted = MutableStateFlow(false)
    private val timeoutCancellationRequested = MutableStateFlow(false)

    suspend operator fun invoke(
        context: Context,
        placement: String?,
        unityAdsShowOptions: UnityAdsShowOptions?,
        unityShowListener: IUnityAdsShowListener?
    ) {
        val startTime = TimeSource.Monotonic.markNow()
        DeviceLog.debug("Unity Ads Show Start for placement $placement")
        showStart()

        val reportShowError = showError(startTime, placement, unityShowListener)

        if (placement == null) {
            reportShowError(REASON_PLACEMENT_NULL, UnityAds.UnityAdsShowError.INVALID_ARGUMENT, MSG_PLACEMENT_NULL, null)
            return
        }

        if (getInitializationState() != InitializationState.INITIALIZED) {
            reportShowError(REASON_NOT_INITIALIZED, UnityAds.UnityAdsShowError.NOT_INITIALIZED, MSG_NOT_INITIALIZED, null)
            return
        }

        val opportunityId = unityAdsShowOptions?.let(::getOpportunityId)
        if (opportunityId == null) {
            reportShowError(REASON_OPPORTUNITY_ID, UnityAds.UnityAdsShowError.INVALID_ARGUMENT, MESSAGE_OPPORTUNITY_ID, null)
            return
        }

        val opportunityIdByteString = UUID.fromString(opportunityId).toByteString()
        val adObject = adRepository.getAd(opportunityIdByteString)
        if (adObject == null) {
            reportShowError(REASON_AD_OBJECT_NOT_FOUND, UnityAds.UnityAdsShowError.INTERNAL_ERROR, MESSAGE_NO_AD_OBJECT + opportunityId, null)
            return
        }

        if (!adObject.adPlayer.scope.isActive) {
            reportShowError(REASON_AD_PLAYER_SCOPE, UnityAds.UnityAdsShowError.INTERNAL_ERROR, MESSAGE_AD_PLAYER_UNAVAILABLE, null)
            return
        }

        val isBanner = adObject.adPlayer is EmbeddableAdPlayer

        if (!isBanner && isFullscreenAdShowing) {
            reportShowError(REASON_ALREADY_SHOWING, UnityAds.UnityAdsShowError.ALREADY_SHOWING, MESSAGE_ALREADY_SHOWING, null)
            return
        }

        if (sessionRepository.nativeConfiguration.featureFlags.opportunityIdPlacementValidation && adObject.placementId != placement) {
            reportShowError(REASON_PLACEMENT_VALIDATION, UnityAds.UnityAdsShowError.INVALID_ARGUMENT, MSG_OPPORTUNITY_AND_PLACEMENT_NOT_MATCHING, null)
            return
        }

        val useTimeout = true // sessionRepository.nativeConfiguration.featureFlags.showTimeoutEnabled
        val timeoutMillis = sessionRepository.nativeConfiguration.adOperations.showTimeoutMs.toLong()

        if (!isBanner) {
            isFullscreenAdShowing = true
        }

        show(context, adObject)
            .timeoutAfter(timeoutMillis) {
                if (hasStarted.value || timeoutCancellationRequested.value) return@timeoutAfter
                sendOperativeError(
                    OperativeEventErrorType.OPERATIVE_EVENT_ERROR_TYPE_TIMEOUT,
                    MESSAGE_OPT_TIMEOUT,
                    adObject
                )
                if (useTimeout) {
                    show.terminate(adObject)
                    reportShowError(REASON_TIMEOUT, UnityAds.UnityAdsShowError.TIMEOUT, MESSAGE_TIMEOUT + placement, null)
                    if (!isBanner) {
                        isFullscreenAdShowing = false
                    }
                }
            }
            .catch {
                reportShowError(REASON_UNCAUGHT_EXCEPTION, UnityAds.UnityAdsShowError.INTERNAL_ERROR, it.message ?: "", null)
                if (!isBanner) {
                    isFullscreenAdShowing = false
                }
            }
            .collect {
                when (it) {
                    is ShowEvent.Started -> showStarted(startTime, placement, unityShowListener)
                    is ShowEvent.Clicked -> showClicked(startTime, placement, unityShowListener)
                    is ShowEvent.Completed -> showCompleted(startTime, placement, it.status, unityShowListener)
                    is ShowEvent.Error -> {
                        sendOperativeError(
                            OperativeEventErrorType.OPERATIVE_EVENT_ERROR_TYPE_UNSPECIFIED,
                            it.message,
                            adObject
                        )
                        reportShowError(it.reason, UnityAds.UnityAdsShowError.INTERNAL_ERROR, it.message, it.errorCode)
                    }

                    is ShowEvent.CancelTimeout -> cancelTimeout(startTime)
                }
            }

        if (!isBanner) {
            isFullscreenAdShowing = false
        }
    }

    private suspend fun sendOperativeError(
        operativeEvent: OperativeEventErrorType,
        operativeMessage: String,
        adObject: AdObject
    ) {
        val errorData = operativeEventErrorData {
            errorType = operativeEvent
            message = operativeMessage
        }
        getOperativeEventApi(
            operativeEventType = OperativeEventType.OPERATIVE_EVENT_TYPE_SHOW_ERROR,
            adObject = adObject,
            additionalEventData = errorData.toByteString()
        )
    }

    private fun cancelTimeout(startTime: TimeMark) {
        timeoutCancellationRequested.value = true
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.SHOW_CANCEL_TIMEOUT,
            value = startTime.elapsedMillis()
        )
    }

    private suspend fun showStarted(startTime: TimeMark, placement: String, unityShowListener: IUnityAdsShowListener?) {
        DeviceLog.debug("Unity Ads Show WV Start for placement $placement")
        hasStarted.value = true
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.SHOW_WV_STARTED,
            value = startTime.elapsedMillis()
        )
        withContext(dispatcher) {
            unityShowListener?.onUnityAdsShowStart(placement)
        }
    }

    private suspend fun showClicked(startTime: TimeMark, placement: String, unityShowListener: IUnityAdsShowListener?) {
        DeviceLog.debug("Unity Ads Show Clicked for placement $placement")
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.SHOW_CLICKED,
            value = startTime.elapsedMillis()
        )
        withContext(dispatcher) {
            unityShowListener?.onUnityAdsShowClick(placement)
        }
    }

    private fun getOpportunityId(unityAdsShowOptions: UnityAdsShowOptions): String? {
        val objectId = unityAdsShowOptions.data?.opt(KEY_OBJECT_ID)?.toString()
        return try {
            UUID.fromString(objectId).toString()
        } catch (t: Throwable) {
            null
        }
    }

    private fun showStart() = sendDiagnosticEvent(event = SendDiagnosticEvent.SHOW_STARTED)

    private suspend fun showCompleted(
        startTime: TimeMark,
        placement: String,
        status: ShowStatus,
        unityShowListener: IUnityAdsShowListener?,
    ) {
        DeviceLog.debug("Unity Ads Show Completed for placement $placement")
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.SHOW_SUCCESS,
            value = startTime.elapsedMillis()
        )
        withContext(dispatcher) {
            unityShowListener?.onUnityAdsShowComplete(placement, status.toUnityAdsShowCompletionState())
        }
    }

    private fun showError(
        startTime: TimeMark,
        placement: String?,
        unityShowListener: IUnityAdsShowListener?
    ): suspend (String, UnityAds.UnityAdsShowError, String, Int?) -> Unit =
        { diagnosticReason: String, reason: UnityAds.UnityAdsShowError, message: String, code: Int? ->
            DeviceLog.debug("Unity Ads Show Failed for placement $placement")
            sendDiagnosticEvent(event = SHOW_FAILURE, value = startTime.elapsedMillis(), tags = getTags(diagnosticReason, code))
            withContext(dispatcher) {
                unityShowListener?.onUnityAdsShowFailure(placement, reason, message)
            }
        }

    private fun getTags(diagnosticReason: String, code: Int?): Map<String, String> {
        return mutableMapOf(
            OPERATION to OperationType.SHOW.toString(),
            REASON to diagnosticReason,
            SHOW_HAS_STARTED to hasStarted.value.toString()
        ).apply {
            code?.let { put(REASON_CODE, it.toString()) }
        }
    }

    companion object {
        const val KEY_OBJECT_ID = "objectId"
        const val MESSAGE_OPPORTUNITY_ID = "No valid opportunity id provided"
        const val MESSAGE_NO_AD_OBJECT = "No ad object found for opportunity id: "
        const val MESSAGE_ALREADY_SHOWING = "Can't show a new ad unit when ad unit is already open"
        const val MESSAGE_TIMEOUT = "[UnityAds] Timeout while trying to show "
        const val MSG_OPPORTUNITY_AND_PLACEMENT_NOT_MATCHING =
            "[UnityAds] Object ID and Placement ID provided does not match previously loaded ad"
        const val MESSAGE_OPT_TIMEOUT = "timeout"
        const val MESSAGE_AD_PLAYER_UNAVAILABLE = "Ad player is unavailable."
    }
}