package com.unity3d.ads.core.domain

import com.unity3d.services.core.network.core.HttpClient
import com.unity3d.services.core.network.model.HttpRequest
import com.unity3d.services.core.network.model.HttpRequest.Companion.DEFAULT_TIMEOUT
import com.unity3d.services.core.network.model.HttpResponse
import com.unity3d.services.core.network.model.RequestType
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import org.json.JSONArray

class AndroidExecuteAdViewerRequest(
    private val ioDispatcher: CoroutineDispatcher,
    private val httpClient: HttpClient,
) : ExecuteAdViewerRequest {

    override suspend operator fun invoke(type: RequestType, parameters: Array<Any>): HttpResponse = withContext(ioDispatcher) {
        val request = createRequest(type, parameters)
        httpClient.execute(request)
    }

    private fun createRequest(type: RequestType, parameters: Array<Any>): HttpRequest {

        val url = parameters.getOrNull(1) as String?
        require(!url.isNullOrEmpty())

        return when (type) {
            RequestType.GET, RequestType.HEAD -> {
                val headers = parameters.getOrNull(2) as JSONArray?
                val connectTimeout = parameters.getOrNull(3) as Int?
                val readTimeout = parameters.getOrNull(4) as Int?

                HttpRequest(
                    baseURL = url,
                    method = type,
                    headers = getHeadersMap(headers),
                    connectTimeout = connectTimeout ?: DEFAULT_TIMEOUT,
                    readTimeout = readTimeout ?: DEFAULT_TIMEOUT,
                )
            }
            RequestType.POST -> {
                val body = parameters.getOrNull(2) as String?
                val headers = parameters.getOrNull(3) as JSONArray?
                val connectTimeout = parameters.getOrNull(4) as Int?
                val readTimeout = parameters.getOrNull(5) as Int?

                HttpRequest(
                    baseURL = url,
                    method = type,
                    headers = getHeadersMap(headers),
                    body = body,
                    connectTimeout = connectTimeout ?: DEFAULT_TIMEOUT,
                    readTimeout = readTimeout ?: DEFAULT_TIMEOUT,
                )
            }
        }

    }

    fun getHeadersMap(headers: JSONArray?): Map<String, List<String>> {

        if (headers == null) return emptyMap()

        val mappedHeaders: MutableMap<String, MutableList<String>> = mutableMapOf()
        for (idx in 0 until headers.length()) {
            val header = headers[idx] as JSONArray
            var valueList = mappedHeaders[header.getString(0)]
            if (valueList == null) {
                valueList = mutableListOf()
            }
            valueList.add(header.getString(1))
            mappedHeaders[header.getString(0)] = valueList
        }
        return mappedHeaders
    }
}