package com.unity3d.ads.core.data.repository

import com.google.protobuf.ByteString
import com.unity3d.ads.core.data.datasource.ByteStringDataSource
import com.unity3d.ads.core.data.model.InitializationState
import com.unity3d.ads.core.data.model.SessionChange
import com.unity3d.ads.core.extensions.toByteString
import com.unity3d.services.UnityAdsConstants
import com.unity3d.services.core.properties.ClientProperties
import com.unity3d.services.core.properties.SdkProperties
import gateway.v1.NativeConfigurationOuterClass.NativeConfiguration
import gateway.v1.NativeConfigurationOuterClass.FeatureFlags
import gateway.v1.SessionCountersOuterClass.SessionCounters
import gateway.v1.copy
import kotlinx.coroutines.flow.*
import java.util.UUID

internal class AndroidSessionRepository(
    private val gatewayCacheDataSource: ByteStringDataSource,
    private val privacyDataSource: ByteStringDataSource,
    private val fsmDataSource: ByteStringDataSource,
    defaultNativeConfiguration: NativeConfiguration
) : SessionRepository {
    private var _onChange = MutableSharedFlow<SessionChange>()
    override val onChange = _onChange.asSharedFlow()

    private var _gameId = MutableStateFlow(value = ClientProperties.getGameId())
    override var gameId
        get() = _gameId.updateAndGet { ClientProperties.getGameId() }
        set(value) = _gameId.update {
            ClientProperties.setGameId(value)
            value
        }

    private val _sessionId = MutableStateFlow(value = UUID.randomUUID().toByteString())
    override val sessionId: ByteString
        get() = _sessionId.value

    private val _isTestModeEnabled = MutableStateFlow(value = SdkProperties.isTestMode())
    override val isTestModeEnabled: Boolean
        get() = _isTestModeEnabled.updateAndGet { SdkProperties.isTestMode() }

    private val _sessionCounters = MutableStateFlow<SessionCounters>(value = SessionCounters.newBuilder().build())
    override var sessionCounters: SessionCounters
        get() = _sessionCounters.value
        set(value) = _sessionCounters.update { value }

    override fun incrementLoadRequestCount() {
        _sessionCounters.update {
            it.copy { loadRequests += 1 }
        }
    }

    override fun incrementLoadRequestAdmCount() {
        _sessionCounters.update {
            it.copy { loadRequestsAdm += 1 }
        }
    }

    private val _sessionToken = MutableStateFlow<ByteString>(value = ByteString.EMPTY)
    override var sessionToken: ByteString
        get() = _sessionToken.value
        set(value) = _sessionToken.update { value }

    private val _currentState = MutableStateFlow<ByteString>(value = ByteString.EMPTY)
    override var gatewayState: ByteString
        get() = _currentState.value
        set(value) = _currentState.update { value }

    private val _sdkConfiguration = MutableStateFlow<NativeConfiguration>(value = defaultNativeConfiguration)
    override var nativeConfiguration: NativeConfiguration
        get() = _sdkConfiguration.value
        set(value) = _sdkConfiguration.update { value }

    private val _gatewayUrl = MutableStateFlow(value = UnityAdsConstants.DefaultUrls.GATEWAY_URL)
    override var gatewayUrl: String
        get() = _gatewayUrl.value
        set(value) = _gatewayUrl.update { value }

    private val _initializationState = MutableStateFlow(value = InitializationState.NOT_INITIALIZED)
    override var initializationState: InitializationState
        get() = _initializationState.value
        set(value) = _initializationState.update { value }

    override val isSdkInitialized: Boolean
        get() = initializationState == InitializationState.INITIALIZED

    override val isDiagnosticsEnabled: Boolean
        get() = nativeConfiguration.diagnosticEvents.enabled

    override val featureFlags: FeatureFlags
        get() = nativeConfiguration.featureFlags

    private val _headerBiddingTokenCounter = MutableStateFlow(value = 0)
    override val headerBiddingTokenCounter: Int
        get() = _headerBiddingTokenCounter.getAndUpdate { it + 1 }

    override suspend fun getGatewayCache(): ByteString = gatewayCacheDataSource.get().data
    override suspend fun setGatewayCache(value: ByteString) = gatewayCacheDataSource.set(value)

    override suspend fun getPrivacy(): ByteString = privacyDataSource.get().data
    override suspend fun setPrivacy(value: ByteString) {
        privacyDataSource.set(value)
        _onChange.emit(SessionChange.UserConsentChange(value))
    }

    override suspend fun getPrivacyFsm(): ByteString = fsmDataSource.get().data
    override suspend fun setPrivacyFsm(value: ByteString) {
        fsmDataSource.set(value)
        _onChange.emit(SessionChange.PrivacyFsmChange(value))
    }

    private val _shouldInitialize = MutableStateFlow(value = true)
    override var shouldInitialize: Boolean
        get() = _shouldInitialize.value
        set(value) = _shouldInitialize.update { value }
}