package com.unity3d.services.core.network.domain

import java.io.File

class CleanupDirectory {
    operator fun invoke(directory: File, sizeLimitMb: Int, ageLimitMs: Long) {
        require(directory.exists() && directory.isDirectory) {
            "Directory does not exist or is not a directory: $directory"
        }

        val cachedFiles = directory.walk().filter(File::isFile)
        val currentSize = cachedFiles.sumOf(File::length)

        val now = System.currentTimeMillis()
        val (staleFiles, remainingFiles) = cachedFiles.partition {
            it.lastModified() + ageLimitMs < now
        }

        // Remove stale files and calculate remaining size
        val sizeAfterStaleRemoval = currentSize - staleFiles.sumOf(File::length)
        staleFiles.forEach(File::delete)

        // If still over limit, remove oldest remaining files until under limit
        val cacheSizeLimitBytes = sizeLimitMb * 1024 * 1024
        if (sizeAfterStaleRemoval > cacheSizeLimitBytes) {
            val additionalFiles = remainingFiles
                .asSequence()
                .sortedBy(File::lastModified)
                .runningFold(sizeAfterStaleRemoval to emptyList<File>()) { (sizeAfterRemoval, files), file ->
                    (sizeAfterRemoval - file.length()) to (files + file)
                }
                .firstOrNull { (size, _) -> size <= cacheSizeLimitBytes }
                // If null, means all files have to be removed to be under limit
                ?.let { (_, selectedFiles) -> selectedFiles } ?: remainingFiles

            additionalFiles.forEach(File::delete)
        }
    }
}