package com.unity3d.services.core.extensions

import kotlinx.coroutines.Deferred
import kotlinx.coroutines.async
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlin.coroutines.cancellation.CancellationException

/**
 * Stops runCatching from swallowing [CancellationException] and messing up structured concurrency
 */
inline fun <R> runSuspendCatching(block: () -> R): Result<R> {
    return try {
        Result.success(block())
    } catch (c: CancellationException) {
        throw c
    } catch (e: Throwable) {
        Result.failure(e)
    }
}

inline fun <R> runReturnSuspendCatching(block: () -> R): Result<R> {
    return runSuspendCatching(block)
        .onSuccess { return Result.success(it) }
        .onFailure { return Result.failure(it) }
}

val deferreds = object : LinkedHashMap<Any, Deferred<*>>(101) {
    override fun removeEldestEntry(eldest: MutableMap.MutableEntry<Any, Deferred<*>>): Boolean {
        return size > 100
    }
}
val mutex = Mutex()

suspend inline fun <T> memoize(
    key: Any,
    crossinline action: suspend () -> T
): T = coroutineScope {
    val deferred = mutex.withLock {
        deferreds.getOrPut(key) { async { action() } }
    }

    @Suppress("UNCHECKED_CAST")
    deferred.await() as T
}
