package com.unity3d.ads.core.domain.events


import com.google.protobuf.kotlin.toByteString
import com.unity3d.ads.core.data.datasource.UniversalRequestDataSource
import com.unity3d.ads.core.data.repository.OperativeEventRepository
import com.unity3d.ads.core.domain.GetRequestPolicy
import com.unity3d.ads.core.domain.GetUniversalRequestForPayLoad
import com.unity3d.ads.core.domain.work.BackgroundWorker
import com.unity3d.ads.core.domain.work.OperativeEventJob
import com.unity3d.ads.core.domain.work.UniversalRequestWorkerData
import gatewayprotocol.v1.UniversalRequestKt
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.getAndUpdate
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.withContext
import java.util.UUID

class OperativeEventObserver(
    private val getUniversalRequestForPayLoad: GetUniversalRequestForPayLoad,
    defaultDispatcher: CoroutineDispatcher,
    private val operativeEventRepository: OperativeEventRepository,
    private val universalRequestDataSource: UniversalRequestDataSource,
    private val backgroundWorker: BackgroundWorker,
    private val universalRequestEventSender: UniversalRequestEventSender,
    private val operativeRequestPolicy: GetRequestPolicy
) {
    private val isRunning = MutableStateFlow(false)

    // We provide an exception handler in the worst case scenario where we can't persist nor even send out the event.
    private val scope = CoroutineScope(defaultDispatcher + CoroutineExceptionHandler { _, _ -> })

    suspend operator fun invoke() = withContext(scope.coroutineContext)  {
        // If we are already collecting events, do nothing
        if (isRunning.getAndUpdate { true }) return@withContext
        operativeEventRepository.operativeEvents.onEach { operativeEventRequest ->
            val payload = UniversalRequestKt.payload {
                operativeEvent = operativeEventRequest
            }
            val fullRequest = getUniversalRequestForPayLoad(payload)

            try {
                val workId = UUID.randomUUID().toString()
                universalRequestDataSource.set(workId, fullRequest.toByteArray().toByteString())

                val universalRequestWorkerData = UniversalRequestWorkerData(workId)
                backgroundWorker<OperativeEventJob>(universalRequestWorkerData)
            }  catch (throwable: Throwable) {
                universalRequestEventSender(fullRequest, operativeRequestPolicy())
            }
        }.launchIn(scope)
    }
}