package com.unity3d.ads.core.data.datasource

import com.unity3d.ads.core.data.model.CacheError
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.model.CacheSource
import com.unity3d.ads.core.data.model.CachedFile
import com.unity3d.ads.core.domain.CreateFile
import com.unity3d.ads.core.domain.GetFileExtensionFromUrl
import com.unity3d.ads.core.domain.HttpClientProvider
import com.unity3d.services.core.network.model.HttpRequest
import com.unity3d.services.core.network.model.isSuccessful
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import okio.Okio
import java.io.File
import java.io.InputStream

class AndroidRemoteCacheDataSource(
    private val ioDispatcher: CoroutineDispatcher,
    private val createFile: CreateFile,
    private val getFileExtensionFromUrl: GetFileExtensionFromUrl,
    private val httpClientProvider: HttpClientProvider
) : CacheDataSource {
    override suspend fun getFile(
        cachePath: File,
        fileName: String,
        url: String?,
        priority: Int?
    ): CacheResult = withContext(ioDispatcher) {
        if (url.isNullOrEmpty()) {
            return@withContext CacheResult.Failure(CacheError.MALFORMED_URL, CacheSource.REMOTE)
        }

        val file = createFile(cachePath, "$fileName.part")
        if (!file.exists()) file.createNewFile()
        val fileSizeBefore = file.length()
        val etagFile = createFile(cachePath, "$fileName.etag")
        val etagValue = etagFile.takeIf(File::exists)?.readText()

        val headers = buildMap {
            if (fileSizeBefore > 0) put("Range", listOf("bytes=$fileSizeBefore-"))
            if (etagValue != null) put("If-Range", listOf(""""$etagValue"""")) // note the usage of quotes inside """ """
        }

        val request = HttpRequest(
            baseURL = url,
            priority = priority ?: Int.MAX_VALUE,
            headers = headers
        )

        val httpClient = httpClientProvider()
        val response = httpClient.execute(request, withInputStream = true)

        if (!response.isSuccessful()) {
            return@withContext CacheResult.Failure(
                CacheError.NETWORK_ERROR,
                CacheSource.REMOTE,
                Exception("Request failed with status code ${response.statusCode}"),
            )
        }

        // Write ETag to the file in case the download gets interrupted for later use. (note the removal of quotes)
        val etagReceived = response.headers["ETag"]?.firstOrNull()?.trim('"') ?: ""
        etagReceived.takeIf(String::isNotEmpty)?.let(etagFile::writeText)

        // If the server returns 200 instead of 206, the file changed - start over
        if (fileSizeBefore > 0 && response.statusCode == 200) {
            file.delete()
            file.createNewFile()
        }

        val body = response.body as? InputStream
            ?: return@withContext CacheResult.Failure(
                CacheError.NETWORK_ERROR,
                CacheSource.REMOTE,
                Exception("Response body is not an InputStream"),
            )

        // Effective total of bytes read from the network
        var totalBytesRead = 0

        runCatching {
            body.use { inputStream ->
                val buffer = ByteArray(8192)
                var bytesRead: Int

                Okio.appendingSink(file).use { fileSink ->
                    Okio.buffer(fileSink).use { bufferedSink ->
                        while (inputStream.read(buffer).also { bytesRead = it } != -1) {
                            // Write to the file new data
                            bufferedSink.write(buffer, 0, bytesRead)
                            bufferedSink.flush()
                            totalBytesRead += bytesRead
                        }
                    }
                }
            }
        }.onFailure {
            return@withContext CacheResult.Failure(
                CacheError.NETWORK_ERROR,
                CacheSource.REMOTE,
                it
            )
        }

        val fileIsComplete = if (response.statusCode == 206) {
            file.length() == (response.contentSize + fileSizeBefore)
        } else if (response.contentSize != -1L) {
            file.length() == response.contentSize
        } else {
            // It means the server did not return a content size.
            file.length() > 0
        }

        if (!fileIsComplete) {
            return@withContext CacheResult.Failure(CacheError.NETWORK_ERROR, CacheSource.REMOTE)
        }

        val finalFile = File(cachePath, fileName)

        runCatching {
            if (finalFile.exists()) {
                check(finalFile.delete()) { "Final file exists and could not be deleted before overwriting"}
            }

            // Rename the file to the final name (without .part), indicating the file is fully cached.
            check(file.renameTo(finalFile)) { "Could not rename temporary file to final file" }

            // Download is complete, Etag is no longer needed
            if (etagFile.exists()) {
                check(etagFile.delete()) { "Could not delete Etag file after successful download" }
            }
        }.onFailure {
            return@withContext CacheResult.Failure(
                CacheError.FILE_STATE_WRONG,
                CacheSource.REMOTE,
                it
            )
        }

        val cachedFile = CachedFile(
            url = url,
            name = fileName,
            file = finalFile,
            extension = getFileExtensionFromUrl(url) ?: "",
            contentLength = totalBytesRead.toLong(),
            protocol = response.protocol,
            priority = priority ?: Int.MAX_VALUE
        )

        return@withContext CacheResult.Success(cachedFile, CacheSource.REMOTE)
    }
}