package com.unity3d.ads.adplayer

import android.os.Build
import android.webkit.RenderProcessGoneDetail
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import android.webkit.WebView
import androidx.webkit.WebViewClientCompat
import com.unity3d.ads.adplayer.model.WebViewClientError
import com.unity3d.ads.adplayer.model.ErrorReason
import com.unity3d.ads.core.domain.GetCachedAsset
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.WEBVIEW_COULD_NOT_HANDLE_INTERCEPTED_URL
import com.unity3d.ads.core.extensions.removeViewFromParent
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update

class AndroidWebViewClient(
    private val getCachedAsset: GetCachedAsset,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
) : WebViewClientCompat() {
    private val loadErrors = MutableStateFlow(listOf<WebViewClientError>())
    private val _onLoadFinished = CompletableDeferred<List<WebViewClientError>>()
    val onLoadFinished: Deferred<List<WebViewClientError>> = _onLoadFinished

    private val _isRenderProcessGone = MutableStateFlow(false)
    val isRenderProcessGone = _isRenderProcessGone.asStateFlow()

    private val webviewType = MutableStateFlow("")

    override fun onPageFinished(view: WebView, url: String) {
        if (url == BLANK_PAGE) {
            loadErrors.update { it + WebViewClientError(url, ErrorReason.REASON_WEB_BLANK) }
        }

        super.onPageFinished(view, url)
        _onLoadFinished.complete(loadErrors.value)
    }

    override fun shouldInterceptRequest(
        view: WebView,
        request: WebResourceRequest
    ): WebResourceResponse? {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) return super.shouldInterceptRequest(view, request)
        val url = request.url ?: return super.shouldInterceptRequest(view, request)

        try {
            val queryWebviewType = runCatching { url.getQueryParameter("webviewType") }.getOrNull()
            if (!queryWebviewType.isNullOrBlank()) {
                webviewType.value = queryWebviewType
            }

            if (url.lastPathSegment == "favicon.ico") {
                return WebResourceResponse("image/png", null, null)
            }

            return getCachedAsset(request.url, webviewType.value)
        } catch (e: Throwable) {
            val tags = e.message?.let { message -> 
                mapOf(REASON to message)
            } ?: emptyMap()
            sendDiagnosticEvent(WEBVIEW_COULD_NOT_HANDLE_INTERCEPTED_URL, tags = tags)
        }

        return super.shouldInterceptRequest(view, request)
    }

    override fun onRenderProcessGone(view: WebView, detail: RenderProcessGoneDetail): Boolean {
        view.removeViewFromParent()
        view.destroy()
        // todo: if during load and show -> rebuild webview to not compromise show
        // todo: add diagnostic for render process gone during load/show

        // If onRenderProcessGone is called before the page is finished loading, complete the deferred with the error,
        // so it is reported as a load error.
        if (!_onLoadFinished.isCompleted) {
            loadErrors.update {
                it + WebViewClientError(
                    view.url.toString(),
                    ErrorReason.REASON_WEBVIEW_RENDER_PROCESS_GONE
                )
            }

            _onLoadFinished.complete(loadErrors.value)
        } else {
            _isRenderProcessGone.value = true
        }

        return true // returning false will kill the app
    }

    companion object {
        const val BLANK_PAGE = "about:blank"
    }
}