package com.unity3d.ads.core.domain.scar

import com.google.protobuf.ByteString
import com.unity3d.ads.TokenConfiguration
import com.unity3d.ads.core.data.manager.ScarManager
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SCAR_SIGNALS_COLLECTION_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SCAR_SIGNALS_COLLECTION_STARTED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SCAR_SIGNALS_COLLECTION_SUCCESS
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SCAR_SIGNALS_UPLOAD_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SCAR_SIGNALS_UPLOAD_SUCCESS
import com.unity3d.ads.core.extensions.elapsedMillis
import com.unity3d.ads.core.extensions.toProtoAdFormat
import gatewayprotocol.v1.AdFormatOuterClass.AdFormat
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

class AndroidFetchSignalsAndSendUseCase(
    private val scope: CoroutineScope,
    private val sessionRepository: SessionRepository,
    private val scarManager: ScarManager,
    private val handleGetTokenRequest: HandleGetTokenRequest,
    private val sendDiagnosticEvent: SendDiagnosticEvent
) : FetchSignalsAndSendUseCase {
    @OptIn(ExperimentalTime::class)
    override suspend fun invoke(tokenNumber: Int, tokenId: ByteString, tokenConfiguration: TokenConfiguration?) {
        scope.launch {
            val startTime = TimeSource.Monotonic.markNow()
            val requestedSignals = getRequestedSignalFormats(tokenConfiguration)
            if (requestedSignals.isEmpty()) { return@launch }
            sendDiagnosticEvent(SCAR_SIGNALS_COLLECTION_STARTED, tokenNumber = tokenNumber)
            val scarSignals = runCatching {
                scarManager.getSignals(requestedSignals)
            }.getOrNull()
            val signalsCollectionEvent = if (scarSignals != null) {
                SCAR_SIGNALS_COLLECTION_SUCCESS
            } else {
                SCAR_SIGNALS_COLLECTION_FAILURE
            }
            sendDiagnosticEvent(
                event = signalsCollectionEvent,
                value = startTime.elapsedMillis(),
                tags = getTags(),
                tokenNumber = tokenNumber
            )
            if (scarSignals == null) {
                return@launch
            }

            val uploadStartTime = TimeSource.Monotonic.markNow()
            sendDiagnosticEvent(SendDiagnosticEvent.SCAR_SIGNALS_UPLOAD_STARTED, tokenNumber = tokenNumber)

            val uploadResponse = handleGetTokenRequest(tokenId, scarSignals)
            val signalsUploadEvent = if (uploadResponse.hasError()) {
                SCAR_SIGNALS_UPLOAD_FAILURE
            } else {
                SCAR_SIGNALS_UPLOAD_SUCCESS
            }

            sendDiagnosticEvent(
                event = signalsUploadEvent,
                value = uploadStartTime.elapsedMillis(),
                tags = getTags(),
                tokenNumber = tokenNumber
            )
        }
    }

    private fun getRequestedSignalFormats(tokenConfiguration: TokenConfiguration?): List<AdFormat> {
        return if (tokenConfiguration == null) {
            sessionRepository.scarEligibleFormats
        } else {
            tokenConfiguration.adFormat.toProtoAdFormat().takeIf {
                sessionRepository.scarEligibleFormats.contains(it)
            }?.let {
                listOf(it)
            } ?: emptyList()
        }
    }

    private fun getTags(): Map<String, String> {
        val tags = mutableMapOf<String, String>()
        val formats = sessionRepository.scarEligibleFormats

        if (formats.contains(AdFormat.AD_FORMAT_BANNER)) {
            tags["banner"] = "true"
        }
        if (formats.contains(AdFormat.AD_FORMAT_REWARDED)) {
            tags["rewarded"] = "true"
        }
        if (formats.contains(AdFormat.AD_FORMAT_INTERSTITIAL)) {
            tags["interstitial"] = "true"
        }

        return tags
    }
}