package com.unity3d.ads.adplayer

import android.os.Bundle
import android.view.KeyEvent
import android.view.ViewGroup
import android.webkit.WebView
import androidx.activity.ComponentActivity
import androidx.lifecycle.lifecycleScope
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_AD_PLAYER_SCOPE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_OPPORTUNITY_ID
import com.unity3d.ads.core.extensions.toBuiltInMap
import com.unity3d.ads.core.extensions.toByteString
import com.unity3d.services.core.di.IServiceComponent
import com.unity3d.services.core.di.inject
import com.unity3d.services.core.domain.ISDKDispatchers
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.ensureActive
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.onSubscription
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.suspendCancellableCoroutine
import org.json.JSONObject
import java.util.UUID
import kotlin.coroutines.cancellation.CancellationException
import kotlin.coroutines.resume

/**
 * An activity that displays a fullscreen webview.
 */
class FullScreenWebViewDisplay : ComponentActivity(), IServiceComponent {
    private var opportunityId = ""
    private var showOptions: Map<String, Any?>? = null
    private val sendDiagnosticEvent by inject<SendDiagnosticEvent>()
    private val adObject by lazy {
        val adRepository by inject<AdRepository>()
        runCatching {
            adRepository.getAd(UUID.fromString(opportunityId).toByteString())
        }.getOrNull()
    }
    private val dispatchers by inject<ISDKDispatchers>()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        opportunityId = intent.getStringExtra("opportunityId") ?: "not_provided"

        sendDiagnosticEvent(SendDiagnosticEvent.SHOW_AD_VIEWER_FULLSCREEN_INTENT_CREATION_STARTS, tags = mapOf(
            "intentOpportunityId" to opportunityId,
        ), adObject = adObject)

        if (opportunityId == "not_provided") {
            setResult(RESULT_CANCELED)
            // Use CoroutineScope for launching coroutine outside lifecycleScope if activity might finish immediately
            CoroutineScope(dispatchers.default).launch {
                AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayError(opportunityId, "Opportunity ID not found"))
            }
            sendDiagnosticEvent(SendDiagnosticEvent.SHOW_AD_VIEWER_FULLSCREEN_INTENT_CREATION_FAILS, tags = mapOf(
                REASON_DEBUG to REASON_OPPORTUNITY_ID,
            ))
            finish()
            return
        }

        val adPlayer = adObject?.adPlayer
        if (adPlayer?.scope?.isActive != true) {
            setResult(RESULT_CANCELED)
            CoroutineScope(dispatchers.default).launch {
                AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayError(opportunityId, "AdPlayer is not active. Could be because show was called while the app was in background."))
            }
            finish()
            sendDiagnosticEvent(SendDiagnosticEvent.SHOW_AD_VIEWER_FULLSCREEN_INTENT_CREATION_FAILS, tags = mapOf(
                REASON_DEBUG to REASON_AD_PLAYER_SCOPE,
            ))
            return
        }

        intent.hasExtra("orientation")
            .takeIf { it }
            ?.let { requestedOrientation = intent.getIntExtra("orientation", -1) }

        showOptions = intent.getStringExtra("showOptions")?.let {
            runCatching { JSONObject(it).toBuiltInMap() }.getOrNull()
        }

        lifecycleScope.launch {
            listenToAdPlayerEvents()
            sendDiagnosticEvent(SendDiagnosticEvent.SHOW_AD_VIEWER_FULLSCREEN_INTENT_CREATION_SUCCESS, adObject = adObject)
        }
    }

    private suspend fun listenToAdPlayerEvents() = suspendCancellableCoroutine { continuation ->
        AndroidFullscreenWebViewAdPlayer.displayMessages
            .onSubscription {
                lifecycleScope.launch(dispatchers.default) {
                    AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.WebViewInstanceRequest(opportunityId))
                    ensureActive()
                    continuation.resume(Unit)
                }
            }
            .filter { it.opportunityId == opportunityId }
            .onEach {
                when (it) {
                    is DisplayMessage.DisplayFinishRequest -> finish()
                    is DisplayMessage.WebViewInstanceResponse -> loadWebView(it.webView)
                    is DisplayMessage.SetOrientation -> requestedOrientation = it.orientation
                    else -> Unit
                }
            }
            .launchIn(lifecycleScope)
    }

    private fun loadWebView(webView: WebView) {
        CoroutineScope(dispatchers.main).launch {
            try {
                (webView.parent as? ViewGroup)?.removeView(webView)
                setContentView(webView)
                CoroutineScope(dispatchers.default).launch {
                    AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayReady(opportunityId, showOptions))
                }
            } catch (error: Throwable) {
                if (error is CancellationException) return@launch

                CoroutineScope(dispatchers.default).launch {
                    AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.DisplayError(opportunityId, "WebView failed to attach to FullScreenWebViewDisplay."))
                }

                val tags = mapOf(REASON to (error.message ?: "Unknown"))
                sendDiagnosticEvent(SendDiagnosticEvent.SHOW_AD_VIEWER_FULLSCREEN_INTENT_FAILED_TO_ATTACH_WEBVIEW, adObject = adObject, tags = tags)
                setResult(RESULT_CANCELED)
                finish()
            }
        }
    }

    override fun onKeyDown(keyCode: Int, event: KeyEvent?): Boolean {
        return keyCode == KeyEvent.KEYCODE_BACK
    }

    override fun onResume() {
        super.onResume()

        CoroutineScope(dispatchers.default).launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.VisibilityChanged(opportunityId, true))
        }
    }

    override fun onPause() {
        super.onPause()

        CoroutineScope(dispatchers.default).launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.VisibilityChanged(opportunityId, false))
        }
    }

    override fun onDestroy() {
        super.onDestroy()

        sendDiagnosticEvent(SendDiagnosticEvent.SHOW_AD_VIEWER_FULLSCREEN_INTENT_DESTROYED, adObject = adObject)
    }

    override fun onWindowFocusChanged(hasFocus: Boolean) {
        super.onWindowFocusChanged(hasFocus)

        CoroutineScope(dispatchers.default).launch {
            AndroidFullscreenWebViewAdPlayer.displayMessages.emit(DisplayMessage.FocusChanged(opportunityId, hasFocus))
        }
    }
}
