package com.unity3d.services

import android.content.Context
import com.unity3d.ads.IUnityAdsInitializationListener
import com.unity3d.services.core.lifecycle.CachedLifecycle
import com.unity3d.services.core.log.DeviceLog
import com.unity3d.services.core.properties.SdkProperties
import com.unity3d.services.core.properties.Session.Default.id

object UnityServices {
    /**
     * Initializes Unity Ads. Unity Ads should be initialized when app starts.
     * @param context Current Android application context of calling app
     * @param gameId Unique identifier for a game, given by Unity Ads admin tools or Unity editor
     * @param testMode If true, only test ads are shown
     * @param initializationListener Listener for IUnityAdsInitializationListener callbacks
     */
    @JvmStatic
    @Synchronized
    fun initialize(
        context: Context?,
        gameId: String?,
        testMode: Boolean,
        initializationListener: IUnityAdsInitializationListener?
    ) {
        DeviceLog.entered()

        SdkProperties.addInitializationListener(initializationListener)
        SdkProperties.setTestMode(testMode)
        SdkProperties.setDebugMode(SdkProperties.getDebugMode())

        val mode = if (testMode) "test mode" else "production mode"
        DeviceLog.info("Initializing Unity Services ${SdkProperties.getVersionName()} (${SdkProperties.getVersionCode()}) with game id $gameId in $mode, session $id")

        CachedLifecycle.register()

        UnityAdsSDK().initialize(gameId)
    }

    @JvmStatic
    val isSupported: Boolean = true

    @JvmStatic
    val isInitialized: Boolean
        get() = SdkProperties.isInitialized()

    @JvmStatic
    val version: String
        get() = SdkProperties.getVersionName()

    @JvmStatic
    var debugMode: Boolean
        /**
         * Get current debug mode status
         *
         * @return If true, debug mode is on. If false, debug mode is off.
         */
        get() = SdkProperties.getDebugMode()
        /**
         * Toggles debug mode on/off
         *
         * @param debugMode If true, debug mode is on and there will be lots of debug output from Unity Services. If false, there will be only some short log messages from Unity Services.
         */
        set(debugMode) {
            SdkProperties.setDebugMode(debugMode)
        }

    enum class UnityServicesError {
        INVALID_ARGUMENT,
        INIT_SANITY_CHECK_FAIL
    }
}