package com.unity3d.ads.core.domain.attribution

import android.adservices.AdServicesState
import android.adservices.measurement.MeasurementManager
import android.annotation.SuppressLint
import android.content.Context
import android.net.Uri
import android.os.OutcomeReceiver
import android.os.ext.SdkExtensions
import androidx.core.net.toUri
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.extensions.toBase64
import com.unity3d.services.core.device.Device
import com.unity3d.services.core.domain.ISDKDispatchers
import kotlinx.coroutines.asExecutor
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

@SuppressLint("NewApi", "MissingPermission")
class AndroidAttribution(
    context: Context,
    private val dispatchers: ISDKDispatchers,
    private val sessionRepository: SessionRepository
) {
    private val measurementManager by lazy {
        getMeasurementManager(context)
    }

    suspend fun isAvailable(): Boolean {
        if (Device.getApiLevel() < 33) {
            return false
        }

        if (SdkExtensions.getExtensionVersion(SdkExtensions.AD_SERVICES) < 4) {
            return false
        }

        if (measurementManager == null) {
            return false
        }

        if (!AdServicesState.isAdServicesStateEnabled()) {
            return false
        }

        return suspendCoroutine { continuation ->
            measurementManager?.getMeasurementApiStatus(
                dispatchers.default.asExecutor(),
                object : OutcomeReceiver<Int, Exception> {
                    override fun onResult(status: Int) {
                        continuation.resume(status == MeasurementManager.MEASUREMENT_API_STATE_ENABLED)
                    }

                    override fun onError(error: Exception) {
                        continuation.resume(false)
                    }
                }) ?: continuation.resume(false)
        }
    }

    suspend fun registerView(url: String, adObject: AdObject): Boolean {
        if (measurementManager == null) {
            return false
        }

        return suspendCoroutine { continuation ->
            measurementManager?.registerSource(
                getUri(url, adObject),
                null,
                dispatchers.default.asExecutor(),
                object : OutcomeReceiver<Any, Exception> {
                    override fun onResult(p0: Any) {
                        continuation.resume(true)
                    }

                    override fun onError(error: Exception) {
                        continuation.resume(false)
                    }
                }) ?: continuation.resume(false)
        }
    }

    suspend fun registerClick(url: String, adObject: AdObject): Boolean {
        if (measurementManager == null) {
            return false
        }

        val lastInputEvent = adObject.adPlayer?.webViewContainer?.lastInputEvent?.value
            ?: return false

        return suspendCoroutine { continuation ->
            measurementManager?.registerSource(
                getUri(url, adObject),
                lastInputEvent,
                dispatchers.default.asExecutor(),
                object : OutcomeReceiver<Any, Exception> {
                    override fun onResult(p0: Any) {
                        continuation.resume(true)
                    }

                    override fun onError(error: Exception) {
                        continuation.resume(false)
                    }
                }) ?: continuation.resume(false)
        }
    }

    private fun getMeasurementManager(context: Context): MeasurementManager? {
        // accessing MeasurementManager without API level and extension version checks can crash old Android devices
        // also accessing SdkExtensions below API level 30 causes exception so check API level first
        if (Device.getApiLevel() < 33) {
            return null
        }

        if (SdkExtensions.getExtensionVersion(SdkExtensions.AD_SERVICES) < 4) {
            return null
        }

        return context.getSystemService(MeasurementManager::class.java)
    }

    private fun getUri(baseUrl: String, adObject: AdObject): Uri {
        return baseUrl.toUri()
            .buildUpon()
            .appendQueryParameter("sessionToken", sessionRepository.sessionToken.toBase64())
            .appendQueryParameter("trackingToken", adObject.trackingToken.toBase64())
            .build()
    }
}