package com.unity3d.services.core.network.core

import androidx.annotation.VisibleForTesting
import com.unity3d.services.core.domain.ISDKDispatchers
import com.unity3d.services.core.log.DeviceLog
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import org.chromium.net.CronetException
import org.chromium.net.UrlRequest
import org.chromium.net.UrlResponseInfo
import java.io.ByteArrayOutputStream
import java.io.IOException
import java.nio.ByteBuffer
import java.nio.channels.Channels

internal abstract class UnityAdsUrlRequestCallback(
    private val dispatchers: ISDKDispatchers,
    val readTimeout: Long,
) : UrlRequest.Callback() {
    private val bytesReceived = ByteArrayOutputStream()
    private val receiveChannel = Channels.newChannel(bytesReceived)

    @VisibleForTesting var task: Job? = null

    override fun onRedirectReceived(request: UrlRequest, info: UrlResponseInfo?, newLocationUrl: String?) {
        request.followRedirect()
    }

    final override fun onResponseStarted(request: UrlRequest, info: UrlResponseInfo) {
        startTimer(request)
        request.read(ByteBuffer.allocateDirect(BYTE_BUFFER_CAPACITY_BYTES))
    }

    final override fun onReadCompleted(
        request: UrlRequest, info: UrlResponseInfo, byteBuffer: ByteBuffer
    ) {
        cancelTimer()

        byteBuffer.flip()
        try {
            receiveChannel.write(byteBuffer)
        } catch (e: IOException) {
            DeviceLog.info("IOException during ByteBuffer read. Details: ", e)
        }
        byteBuffer.clear()
        startTimer(request)
        request.read(byteBuffer)
    }

    final override fun onSucceeded(request: UrlRequest, info: UrlResponseInfo) {
        cancelTimer()

        val bodyBytes = bytesReceived.toByteArray()
        onSucceeded(request, info, bodyBytes)
    }

    override fun onCanceled(request: UrlRequest?, info: UrlResponseInfo?) {
        super.onCanceled(request, info)

        cancelTimer()
    }

    override fun onFailed(request: UrlRequest?, info: UrlResponseInfo?, error: CronetException?) {
        cancelTimer()
    }

    abstract fun onSucceeded(request: UrlRequest, info: UrlResponseInfo, bodyBytes: ByteArray)

    fun startTimer(request: UrlRequest) {
        cancelTimer()

        task = CoroutineScope(dispatchers.io).launch {
            delay(readTimeout)
            request.cancel()
        }
    }

    private fun cancelTimer() {
        task?.cancel()
        task = null
    }

    companion object {
        private const val BYTE_BUFFER_CAPACITY_BYTES = 16 * 1024
    }
}