package com.unity3d.ads.core.domain.scar

import com.unity3d.scar.adapter.common.GMAEvent
import com.unity3d.services.banners.bridge.BannerBridge.BannerEvent
import com.unity3d.services.core.webview.WebViewEventCategory
import com.unity3d.services.core.webview.bridge.IEventSender
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.launch

class CommonScarEventReceiver(
    private val scope: CoroutineScope
) : IEventSender {
    private val _versionFlow = MutableSharedFlow<String?>()
    val versionFlow = _versionFlow.asSharedFlow()

    private val _gmaEventFlow = MutableSharedFlow<GmaEventData>()
    val gmaEventFlow = _gmaEventFlow.asSharedFlow()

    override fun sendEvent(
        eventCategory: Enum<*>,
        eventId: Enum<*>,
        vararg params: Any
    ): Boolean {
        // Parsing GMA events
        if (eventCategory !in setOf(WebViewEventCategory.INIT_GMA, WebViewEventCategory.GMA, WebViewEventCategory.BANNER)) {
            return false
        }
        scope.launch {
            when (eventId) {
                GMAEvent.VERSION -> {
                    params.let {
                        val version = params[0] as String?
                        _versionFlow.emit(version.takeIf { version != "0.0.0" })
                    }
                }

                GMAEvent.AD_LOADED -> {
                    params.let {
                        _gmaEventFlow.emit(
                            GmaEventData(
                                GMAEvent.AD_LOADED,
                                placementId = params[0] as String
                            )
                        )
                    }
                }

                GMAEvent.LOAD_ERROR -> {
                    params.let {
                        _gmaEventFlow.emit(
                            GmaEventData(
                                GMAEvent.LOAD_ERROR,
                                placementId = params[0] as String,
                                queryId = params[1] as String,
                                errorMessage = params[2] as String,
                                errorCode = params[3] as Int
                            )
                        )
                    }
                }

                else -> {
                    if (eventId is GMAEvent) {
                        _gmaEventFlow.emit(
                            GmaEventData(
                                eventId
                            )
                        )
                    }
                    if (eventId is BannerEvent) {
                        _gmaEventFlow.emit(
                            GmaEventData(
                                gmaEvent = GMAEvent.BANNER,
                                bannerEvent = eventId,
                                opportunityId = params[0] as String
                            )
                        )
                    }
                }
            }
        }
        return true
    }

    override fun canSend(): Boolean {
        return true
    }
}