package com.unity3d.ads.core.data.repository

import android.content.Context
import com.unity3d.ads.core.data.datasource.CacheDataSource
import com.unity3d.ads.core.data.model.CacheError
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.model.CacheSource
import com.unity3d.ads.core.data.model.CachedFile
import com.unity3d.ads.core.domain.CreateFile
import com.unity3d.ads.core.domain.GetCacheDirectory
import com.unity3d.ads.core.extensions.getDirectorySize
import com.unity3d.ads.core.extensions.getSHA256Hash
import com.unity3d.services.UnityAdsConstants.DefaultUrls.CACHE_DIR_NAME
import com.unity3d.services.core.extensions.memoize
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.NonCancellable
import kotlinx.coroutines.plus
import kotlinx.coroutines.withContext
import org.json.JSONArray


import java.io.File
import java.io.IOException
import java.util.concurrent.ConcurrentHashMap

class AndroidCacheRepository(
    ioDispatcher: CoroutineDispatcher,
    private val getCacheDirectory: GetCacheDirectory,
    private val localCacheDataSource: CacheDataSource,
    private val remoteCacheDataSource: CacheDataSource,
    private val context: Context
) : CacheRepository {
    private val scope = CoroutineScope(ioDispatcher) + CoroutineName("CacheRepository") + NonCancellable
    // cachedFiles fileName to CachedFile
    val cachedFiles = ConcurrentHashMap<String, CachedFile>()
    // neededFiles fileName to ObjectID <String, Map<ObjectIds>>
    val neededFiles = ConcurrentHashMap<String, MutableSet<String>>()
    private val cacheDir: File = initCacheDir()

    override suspend fun getFile(
        url: String,
        objectId: String,
        headers: JSONArray?,
        priority: Int
    ): CacheResult = withContext(scope.coroutineContext) {
        val filename = getFilename(url)

        // Check if file is already in cache
        val localFile = localCacheDataSource.getFile(cacheDir, filename, url, priority)
        if (localFile is CacheResult.Success) {
            val cachedFile = localFile.cachedFile.copy(objectId = objectId)
            addFileToCache(cachedFile)
            return@withContext localFile
        }

        // Download the file
        val fileResult = remoteCacheDataSource.getFile(cacheDir, filename, url, priority)
        if (fileResult is CacheResult.Success) {
            val cachedFile = fileResult.cachedFile.copy(objectId = objectId)
            addFileToCache(cachedFile)
        }

        return@withContext fileResult
    }

    override fun retrieveFile(fileName: String): CacheResult {
        val cachedFile = cachedFiles[fileName]
        return if (cachedFile != null) {
            CacheResult.Success(cachedFile, CacheSource.LOCAL)
        } else {
            CacheResult.Failure(CacheError.FILE_NOT_FOUND, CacheSource.LOCAL)
        }
    }

    override fun removeFile(cachedFile: CachedFile): Boolean {
        cachedFiles.remove(cachedFile.name)
        neededFiles[cachedFile.name]?.remove(cachedFile.objectId)
        return cachedFile.file?.takeIf(File::exists)?.delete() ?: false
    }

    override suspend fun doesFileExist(fileName: String): Boolean = cachedFiles.containsKey(fileName)

    fun getFilename(url: String): String = url.getSHA256Hash()

    override suspend fun clearCache() {
        withContext(scope.coroutineContext) {
            cacheDir.listFiles()?.forEach { it.delete() }
        }
    }

    override suspend fun getCacheSize(): Long = withContext(scope.coroutineContext) {
        cacheDir.getDirectorySize()
    }

    private fun initCacheDir(): File {
        val dir = getCacheDirectory(getCacheDirBase(), getCacheDirPath())
        dir.mkdirs()
        return dir
    }

    private fun getCacheDirBase(): File = context.cacheDir

    private fun getCacheDirPath(): String = CACHE_DIR_NAME

    private fun addFileToCache(cachedFile: CachedFile) {
        cachedFiles[cachedFile.name] = cachedFile

        val observers = neededFiles[cachedFile.name] ?: mutableSetOf()
        observers.add(cachedFile.objectId)
        neededFiles[cachedFile.name] = observers
    }
}