package com.unity3d.ads.adplayer

import android.view.ViewGroup
import android.webkit.JavascriptInterface
import android.webkit.WebView
import com.unity3d.ads.adplayer.model.ErrorReason
import com.unity3d.ads.adplayer.model.WebViewBridgeInterface
import com.unity3d.ads.adplayer.model.WebViewClientError
import com.unity3d.ads.core.domain.SendWebViewClientErrorDiagnostics
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach

class AndroidWebViewContainer(
    val webView: WebView,
    private val webViewClient: AndroidWebViewClient,
    private val sendWebViewClientErrorDiagnostics: SendWebViewClientErrorDiagnostics,
    mainDispatcher: CoroutineDispatcher,
    defaultDispatcher: CoroutineDispatcher,
    adPlayerScope: CoroutineScope,
) : WebViewContainer {
    val scope = adPlayerScope + mainDispatcher + CoroutineName("AndroidWebViewContainer")

    init {
        webViewClient.isRenderProcessGone
            .filter { it }
            .onEach { onRenderProcessGone() }
            .launchIn(scope + defaultDispatcher)
    }

    private suspend fun onRenderProcessGone() {
        destroy()
        sendWebViewClientErrorDiagnostics(
            listOf(
                WebViewClientError(
                    "Render process gone",
                    ErrorReason.REASON_WEBVIEW_RENDER_PROCESS_GONE
                )
            )
        )
    }

    override suspend fun loadUrl(url: String) {
        withContext(scope.coroutineContext) {
            webView.loadUrl(url)
        }

        val loadResult = webViewClient.onLoadFinished.await()
        if (loadResult.isNotEmpty()) {
            destroy()
            sendWebViewClientErrorDiagnostics(loadResult)
            throw LoadWebViewError(loadResult)
        }
    }

    override suspend fun evaluateJavascript(script: String) {
        try {
            withContext(scope.coroutineContext) {
                webView.evaluateJavascript("javascript:$script", null)
            }
        } catch (e: CancellationException) {
            // ignore cancellation exception from the coroutine scope, any other exception should be rethrown
        }
    }

    override suspend fun addJavascriptInterface(webViewBridgeInterface: WebViewBridge, name: String) {
        try {
            withContext(scope.coroutineContext) {
                val wrapper = object : WebViewBridgeInterface {
                    @JavascriptInterface
                    override fun handleInvocation(message: String) = webViewBridgeInterface.handleInvocation(message)

                    @JavascriptInterface
                    override fun handleCallback(callbackId: String, callbackStatus: String, rawParameters: String) =
                        webViewBridgeInterface.handleCallback(callbackId, callbackStatus, rawParameters)
                }
                webView.addJavascriptInterface(wrapper, name)
            }
        } catch (e: CancellationException) {
            // ignore cancellation exception from the coroutine scope, any other exception should be rethrown
        }

    }

    override suspend fun destroy() {
        withContext(scope.coroutineContext + NonCancellable) {
            (webView.parent as? ViewGroup)?.removeView(webView)
            webView.destroy()
        }

        scope.cancel()
    }
}