package com.unity3d.ads.core.domain

import android.content.Context
import android.util.Base64
import com.google.protobuf.ByteString
import com.unity3d.ads.UnityAds
import com.unity3d.ads.UnityAdsLoadOptions
import com.unity3d.ads.adplayer.*
import com.unity3d.ads.adplayer.AndroidFullscreenWebViewAdPlayer
import com.unity3d.ads.adplayer.WebViewAdPlayer
import com.unity3d.ads.adplayer.model.LoadEvent
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.model.LoadResult
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_COMMUNICATION_FAILURE
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_NO_FILL
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.data.repository.CampaignRepository
import com.unity3d.ads.core.data.repository.DeviceInfoRepository
import com.unity3d.ads.core.data.repository.OpenMeasurementRepository
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_AD_VIEWER
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_GATEWAY
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_INVALID_ENTRY_POINT
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_NO_FILL
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_NO_WEBVIEW_ENTRY_POINT
import com.unity3d.ads.core.domain.events.GetOperativeEventApi
import com.unity3d.ads.core.extensions.fromBase64
import com.unity3d.ads.core.extensions.toBase64
import com.unity3d.ads.core.extensions.toISO8859String
import com.unity3d.ads.core.extensions.toUUID
import com.unity3d.services.UnityAdsConstants
import com.unity3d.services.UnityAdsConstants.Messages.MSG_INTERNAL_ERROR
import com.unity3d.services.banners.BannerViewCache
import com.unity3d.services.core.di.ServiceProvider.DEFAULT_DISPATCHER
import com.unity3d.services.core.properties.SdkProperties
import gateway.v1.*
import gateway.v1.AdResponseOuterClass.AdResponse
import gateway.v1.DiagnosticEventRequestOuterClass.DiagnosticAdType
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.single
import kotlinx.coroutines.withContext
import org.koin.core.annotation.Factory
import org.koin.core.annotation.Named
import java.net.URI

@Factory
internal class HandleGatewayAndroidAdResponse(
    private val adRepository: AdRepository,
    private val getWebViewContainerUseCase: AndroidGetWebViewContainerUseCase,
    private val getWebViewBridge: GetWebViewBridgeUseCase,
    @Named(DEFAULT_DISPATCHER) private val defaultDispatcher: CoroutineDispatcher,
    private val deviceInfoRepository: DeviceInfoRepository,
    private val getHandleInvocationsFromAdViewer: HandleInvocationsFromAdViewer,
    private val sessionRepository: SessionRepository,
    private val campaignRepository: CampaignRepository,
    private val executeAdViewerRequest: ExecuteAdViewerRequest,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val getOperativeEventApi: GetOperativeEventApi,
    private val getLatestWebViewConfiguration: GetLatestWebViewConfiguration,
    private val adPlayerScope: AdPlayerScope,
    private val openMeasurementRepository: OpenMeasurementRepository,
) : HandleGatewayAdResponse {
    override suspend fun invoke(
        loadOptions: UnityAdsLoadOptions,
        opportunityId: ByteString,
        response: AdResponse,
        context: Context,
        placementId: String,
        adType: DiagnosticAdType
    ): LoadResult {
        var adPlayer: AdPlayer? = null

        try {
            if (response.hasError()) {
                return LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.INTERNAL_ERROR,
                    message = MSG_COMMUNICATION_FAILURE,
                    reason = REASON_GATEWAY,
                    reasonDebug = response.error.errorText
                )
            }

            if (response.adData.isEmpty) {
                return LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.NO_FILL,
                    message = MSG_NO_FILL,
                    reason = REASON_NO_FILL
                )
            }

            val webviewConfiguration = getLatestWebViewConfiguration(
                receivedEntryPoint = response.webviewConfiguration.entryPoint,
                receivedVersion = response.webviewConfiguration.version,
                receivedAdditionalFiles = response.webviewConfiguration.additionalFilesList
            )

            if (webviewConfiguration.entryPoint.isEmpty()) {
                return LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.INTERNAL_ERROR,
                    message = MSG_COMMUNICATION_FAILURE,
                    reason = REASON_NO_WEBVIEW_ENTRY_POINT
                )
            }

            val selectedUrl = SdkProperties.getConfigUrl().takeIf { it.endsWith(".html") }
                ?: webviewConfiguration.entryPoint

            val url = try {
                URI(selectedUrl)
            } catch (t: Throwable) {
                return LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.INTERNAL_ERROR,
                    message = MSG_COMMUNICATION_FAILURE,
                    reason = REASON_INVALID_ENTRY_POINT,
                    reasonDebug = selectedUrl
                )
            }
            val query = UnityAdsConstants.DefaultUrls.AD_PLAYER_QUERY_PARAMS + url.query
            val webViewUrl = selectedUrl.substringBeforeLast("?") + query
            val base64ImpressionConfiguration = Base64.encodeToString(
                response.impressionConfiguration.toByteArray(),
                Base64.NO_WRAP
            )
            val webviewContainer = getWebViewContainerUseCase(adPlayerScope)
            val webviewBridge = getWebViewBridge(webviewContainer, adPlayerScope)
            val webViewAdPlayer = WebViewAdPlayer(
                webviewBridge,
                deviceInfoRepository,
                sessionRepository,
                executeAdViewerRequest,
                defaultDispatcher,
                sendDiagnosticEvent,
                webviewContainer,
                adPlayerScope,
            )

            // If we have a bannerView, it means load has been called via [BannerView.load()]
            val bannerView =
                BannerViewCache.getInstance().getBannerView(opportunityId.toUUID().toString())

            adPlayer = if (bannerView == null)
                AndroidFullscreenWebViewAdPlayer(
                    webViewAdPlayer = webViewAdPlayer,
                    webViewContainer = webviewContainer,
                    opportunityId = opportunityId.toISO8859String(),
                    deviceInfoRepository = deviceInfoRepository,
                    sessionRepository = sessionRepository,
                    openMeasurementRepository = openMeasurementRepository,
                ) else
                AndroidEmbeddableWebViewAdPlayer(
                    webViewAdPlayer = webViewAdPlayer,
                    webViewContainer = webviewContainer,
                    opportunityId = opportunityId.toISO8859String(),
                    openMeasurementRepository = openMeasurementRepository,
                )

            deviceInfoRepository.allowedPii
                .onEach { webViewAdPlayer.onAllowedPiiChange(it.toByteArray()) }
                .launchIn(webViewAdPlayer.scope)

            webViewAdPlayer.updateCampaignState.onEach { (data, dataVersion) ->
                val state = campaignRepository.getCampaign(opportunityId)
                    ?.copy {
                        this.data = data.toString(Charsets.ISO_8859_1).fromBase64()
                        this.dataVersion = dataVersion
                    } ?: campaign {
                    this.data = data.toString(Charsets.ISO_8859_1).fromBase64()
                    this.dataVersion = dataVersion
                    this.placementId = placementId
                    this.impressionOpportunityId = opportunityId
                }

                campaignRepository.setCampaign(opportunityId, state)
            }.launchIn(webViewAdPlayer.scope)

            val adObject = AdObject(
                opportunityId = opportunityId,
                placementId = placementId,
                trackingToken = response.trackingToken,
                adPlayer = adPlayer,
                loadOptions = loadOptions,
                adType = adType
            )

            sendDiagnosticEvent(event = SendDiagnosticEvent.LOAD_STARTED_AD_VIEWER, adObject = adObject)

            getHandleInvocationsFromAdViewer(
                onInvocations = webviewBridge.onInvocation,
                adData = response.adData.toBase64(),
                impressionConfig = base64ImpressionConfiguration,
                adDataRefreshToken = response.adDataRefreshToken.toBase64(),
                adObject = adObject,
                onSubscription = { webviewContainer.loadUrl(webViewUrl) }
            ).launchIn(webViewAdPlayer.scope)

            val loadEvent = withContext(webViewAdPlayer.scope.coroutineContext) {
                webViewAdPlayer.onLoadEvent.single()
            }
            if (loadEvent is LoadEvent.Error) {
                withContext(NonCancellable) {
                    cleanup(Error(loadEvent.message), opportunityId, response, adPlayer)
                }
                return LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.INTERNAL_ERROR,
                    message = MSG_INTERNAL_ERROR,
                    reason = REASON_AD_VIEWER,
                    reasonDebug = loadEvent.message
                )
            }

            campaignRepository.setLoadTimestamp(opportunityId)
            adRepository.addAd(opportunityId, adObject)

            return LoadResult.Success(adObject)
        } catch (t: CancellationException) {
            withContext(NonCancellable) {
                cleanup(t, opportunityId, response, adPlayer)
            }
            throw t.cause ?: t
        }
    }

    private suspend fun cleanup(
        t: Throwable,
        opportunityId: ByteString,
        response: AdResponse,
        adPlayer: AdPlayer?
    ) {
        val operativeEventErrorData = operativeEventErrorData {
            errorType =
                OperativeEventRequestOuterClass.OperativeEventErrorType.OPERATIVE_EVENT_ERROR_TYPE_UNSPECIFIED
            message = t.cause?.message ?: t.message ?: ""
        }
        getOperativeEventApi(
            operativeEventType = OperativeEventRequestOuterClass.OperativeEventType.OPERATIVE_EVENT_TYPE_LOAD_ERROR,
            opportunityId = opportunityId,
            trackingToken = response.trackingToken,
            additionalEventData = operativeEventErrorData.toByteString()
        )
        adPlayer?.destroy()
    }
}