package com.unity3d.ads.core.data.manager

import com.google.protobuf.Timestamp
import com.unity3d.ads.core.data.datasource.ByteStringDataSource
import com.unity3d.ads.core.data.repository.TransactionEventRepository
import com.unity3d.ads.core.domain.events.GetTransactionData
import com.unity3d.ads.core.domain.events.GetTransactionRequest
import com.unity3d.services.core.di.ServiceProvider.DATA_STORE_IAP_TRANSACTION
import com.unity3d.services.core.di.ServiceProvider.NAMED_TRANSACTION_SCOPE
import com.unity3d.services.store.StoreMonitor
import com.unity3d.services.store.gpbl.BillingResultResponseCode
import com.unity3d.services.store.gpbl.bridges.BillingResultBridge
import com.unity3d.services.store.gpbl.bridges.PurchaseBridge
import com.unity3d.services.store.gpbl.listeners.BillingInitializationListener
import gateway.v1.TransactionEventRequestOuterClass.TransactionData
import kotlinx.coroutines.CompletableDeferred
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.awaitAll
import kotlinx.coroutines.launch
import org.koin.core.annotation.Named
import org.koin.core.annotation.Single
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException
import kotlin.coroutines.suspendCoroutine

@Single
// Deals with querying purchases from the Google Play Billing Library.
class TransactionEventManager(
    @Named(NAMED_TRANSACTION_SCOPE)
    private val scope: CoroutineScope,
    private val storeMonitor: StoreMonitor,
    private val getTransactionData: GetTransactionData,
    private val getTransactionRequest: GetTransactionRequest,
    private val transactionEventRepository: TransactionEventRepository,
    @Named(DATA_STORE_IAP_TRANSACTION)
    private val iapTransactionStore: ByteStringDataSource
) {
    operator fun invoke() {
        scope.launch {
            if (!storeMonitor.isInitialized) {
                suspendCoroutine {
                    storeMonitor.initialize(object : BillingInitializationListener {
                        override fun onIsAlreadyInitialized() {
                            it.resume(Unit)
                        }

                        override fun onBillingSetupFinished(billingResult: BillingResultBridge) {
                            if (billingResult.responseCode != BillingResultResponseCode.OK) {
                                it.resumeWithException(Exception("Billing setup failed"))
                                return
                            }
                            it.resume(Unit)
                        }

                        override fun onBillingServiceDisconnected() {
                            it.resumeWithException(Exception("Billing service disconnected"))
                        }

                        override fun onPurchaseUpdated(
                            billingResult: BillingResultBridge,
                            purchases: List<PurchaseBridge>?
                        ) {
                            onPurchasesReceived(billingResult, purchases)
                        }
                    })
                }
            }

            storeMonitor.getPurchases(dummyOperationId, INAPP) { billingResult, purchases ->
                onPurchasesReceived(billingResult, purchases)
            }
        }

    }

    private fun onPurchasesReceived(billingResult: BillingResultBridge, purchases: List<PurchaseBridge>?) {
        if (billingResult.responseCode != BillingResultResponseCode.OK || purchases.isNullOrEmpty()) return
        scope.launch {
            val transactionDataList = mutableListOf<TransactionData>()
            val deferredPurchaseList = purchases.map { CompletableDeferred<Unit>() }
            purchases.forEachIndexed { index, purchase ->
                val purchaseTime = purchase.originalJson["purchaseTime"] as Long // Since Epoch, in ms
                val lastIapTransactionSent = Timestamp.parseFrom(iapTransactionStore.get().data)

                if (lastIapTransactionSent.seconds < purchaseTime / 1000f) {
                    storeMonitor.getSkuDetails(dummyOperationId, INAPP, listOf(purchase.originalJson["productId"] as String)) { billingResult, skuDetailsList ->
                        if (skuDetailsList == null || billingResult.responseCode != BillingResultResponseCode.OK) {
                            deferredPurchaseList[index].complete(Unit)
                            return@getSkuDetails
                        }
                        transactionDataList.addAll(skuDetailsList.map { getTransactionData(purchase, it) })
                        deferredPurchaseList[index].complete(Unit)
                    }
                } else {
                    deferredPurchaseList[index].complete(Unit)
                }
            }

            // Wait for all purchases to complete
            awaitAll(*deferredPurchaseList.toTypedArray())

            if (transactionDataList.isNotEmpty()) {
                val transactionRequest = getTransactionRequest(transactionDataList)
                transactionEventRepository.addTransactionEvent(transactionRequest)
            }
        }
    }

    companion object {
        private const val dummyOperationId = 42; // Operation ID not used outside of WebView context.
        private const val INAPP = "inapp"
    }
}