package com.unity3d.ads.gatewayclient

import com.google.protobuf.InvalidProtocolBufferException
import com.unity3d.ads.core.data.model.OperationType
import com.unity3d.ads.core.data.model.exception.UnityAdsNetworkException
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.HandleGatewayUniversalResponse
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.NETWORK_CLIENT
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.NETWORK_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.NETWORK_SUCCESS
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.OPERATION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.PROTOCOL
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_CODE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.RETRIES
import com.unity3d.ads.core.extensions.elapsedMillis
import com.unity3d.services.UnityAdsConstants
import com.unity3d.services.core.log.DeviceLog
import gateway.v1.*
import com.unity3d.services.core.network.core.HttpClient
import com.unity3d.services.core.network.model.HttpRequest
import com.unity3d.services.core.network.model.HttpResponse
import com.unity3d.services.core.network.model.RequestType
import com.unity3d.services.core.network.model.isSuccessful
import com.unity3d.services.core.network.model.toHttpResponse
import kotlinx.coroutines.delay
import org.koin.core.annotation.Single
import kotlin.math.pow
import kotlin.random.Random
import kotlin.time.ExperimentalTime
import kotlin.time.TimeMark
import kotlin.time.TimeSource

@OptIn(ExperimentalTime::class)
@Single
class CommonGatewayClient(
    private val httpClient: HttpClient,
    private val handleGatewayUniversalResponse: HandleGatewayUniversalResponse,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val sessionRepository: SessionRepository,
): GatewayClient {
    companion object {
        const val RETRY_ATTEMPT_HEADER = "X-RETRY-ATTEMPT"
        const val CODE_400 = 400
        const val CODE_599 = 599
    }

    private fun shouldRetry(responseCode: Int): Boolean {
        return responseCode in CODE_400..CODE_599
    }

    override suspend fun request(
        url: String,
        request: UniversalRequestOuterClass.UniversalRequest,
        requestPolicy: RequestPolicy,
        operationType: OperationType
    ): UniversalResponseOuterClass.UniversalResponse {
        var retryCount = 0
        var retryDuration = 0L

        val gatewayUrl = if (url != UnityAdsConstants.DefaultUrls.GATEWAY_URL) {
            url
        } else {
            sessionRepository.gatewayUrl
        }

        do {
            val headers = buildMap {
                put("Content-Type", listOf("application/x-protobuf"))
                if (retryCount > 0) {
                    put(RETRY_ATTEMPT_HEADER, listOf(retryCount.toString()))
                }
            }

            val httpRequest = HttpRequest(
                baseURL = gatewayUrl,
                method = RequestType.POST,
                body = request.toByteArray(),
                headers = headers,
                connectTimeout = requestPolicy.connectTimeout,
                readTimeout = requestPolicy.readTimeout,
                writeTimeout = requestPolicy.writeTimeout,
                isProtobuf = true
            )


            var httpResponse: HttpResponse
            val startTime = TimeSource.Monotonic.markNow()
            try {
                httpResponse = httpClient.execute(httpRequest)
                sendNetworkSuccessDiagnosticEvent(httpResponse, retryCount, operationType, startTime)
            } catch (e: UnityAdsNetworkException) {
                sendNetworkErrorDiagnosticEvent(e, retryCount, operationType, startTime)
                httpResponse = e.toHttpResponse()
            }

            if (!shouldRetry(httpResponse.statusCode)) {
                if (httpResponse.isSuccessful()) {
                    return getUniversalResponse(httpResponse)
                        .also { handleGatewayUniversalResponse(it) }
                } else {
                    throw UnityAdsNetworkException("Gateway request failed after $retryCount retries and duration: ${retryDuration}ms")
                }
            }
            val currentDelayTime = calculateDelayTime(requestPolicy, retryCount)
            retryDuration += currentDelayTime
            delay(currentDelayTime)

            retryCount++
        } while (retryDuration <= requestPolicy.maxDuration)

        throw UnityAdsNetworkException("Gateway request failed after $retryCount retries and duration: ${retryDuration}ms")
    }

    private fun sendNetworkErrorDiagnosticEvent(
        e: UnityAdsNetworkException,
        retryCount: Int,
        operationType: OperationType,
        startTime: TimeMark,
    ) {
       if (operationType == OperationType.UNIVERSAL_EVENT) return

       val tags = mutableMapOf(
            OPERATION to operationType.toString(),
            RETRIES to retryCount.toString(),
            PROTOCOL to e.protocol.toString(),
            NETWORK_CLIENT to e.client.toString(),
            REASON_CODE to e.code.toString(),
            REASON_DEBUG to e.message
       )
        sendDiagnosticEvent(NETWORK_FAILURE, startTime.elapsedMillis(), tags = tags)
    }

    private fun sendNetworkSuccessDiagnosticEvent(
        httpResponse: HttpResponse,
        retryCount: Int,
        operationType: OperationType,
        startTime: TimeMark,
    ) {
        if (operationType == OperationType.UNIVERSAL_EVENT) return

        val tags = mutableMapOf(
            OPERATION to operationType.toString(),
            RETRIES to retryCount.toString(),
            PROTOCOL to httpResponse.protocol,
            NETWORK_CLIENT to httpResponse.client,
            REASON_CODE to httpResponse.statusCode.toString(),
        )
        sendDiagnosticEvent(NETWORK_SUCCESS, startTime.elapsedMillis(), tags = tags)
    }

    private fun getUniversalResponse(response: HttpResponse): UniversalResponseOuterClass.UniversalResponse {
        try {
            val responseBody = response.body
            if (responseBody is ByteArray) {
                return UniversalResponseOuterClass.UniversalResponse.parseFrom(responseBody)
            }
            if (responseBody is String) {
                return UniversalResponseOuterClass.UniversalResponse.parseFrom(
                    responseBody.toByteArray(Charsets.ISO_8859_1)
                )
            }
            throw InvalidProtocolBufferException("Could not parse response from gateway service")

        } catch (e: InvalidProtocolBufferException) {
            DeviceLog.debug("Failed to parse response from gateway service with exception: %s", e.localizedMessage)
            return universalResponse {
                error = error {
                    errorText = "ERROR: Could not parse response from gateway service"
                }
            }
        }
    }

    private fun calculateDelayTime(requestPolicy: RequestPolicy, retryCount: Int): Long {
        val retryWaitTime = calculateExponentialBackoff(requestPolicy.retryWaitBase, retryCount)
        val jitter = calculateJitter(retryWaitTime, requestPolicy.retryJitterPct)
        return retryWaitTime + jitter;
    }

    // TODO Add different variations of backoff policies
    private fun calculateExponentialBackoff(retryWaitBase: Int, retryCount: Int): Long {
        return (retryWaitBase * 2.0.pow(retryCount).toLong())
    }

    private fun calculateJitter(retryWaitTime: Long, retryJitterPct: Float): Long {
        val jitterRange = (retryWaitTime * retryJitterPct).toLong()
        return Random.nextLong(-jitterRange, jitterRange + 1)
    }
}