/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingRequest;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingResponse;
import org.elasticsearch.action.admin.indices.template.put.PutIndexTemplateRequest;
import org.elasticsearch.action.admin.indices.template.put.PutIndexTemplateResponse;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.IndexTemplateMetaData;
import org.elasticsearch.cluster.metadata.MappingMetaData;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.collect.ImmutableOpenMap;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.compress.CompressedXContent;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.xcontent.XContent;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.common.xcontent.json.JsonXContent;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.log4j.Logger;
import org.elasticsearch.log4j.message.ParameterizedMessage;
import org.elasticsearch.xpack.security.InternalClient;
import org.elasticsearch.xpack.security.authc.esnative.NativeRealmMigrator;
import org.elasticsearch.xpack.template.TemplateUtils;

public class SecurityTemplateService
extends AbstractComponent
implements ClusterStateListener {
    public static final String SECURITY_INDEX_NAME = ".security";
    public static final String SECURITY_TEMPLATE_NAME = "security-index-template";
    private static final String SECURITY_VERSION_STRING = "security-version";
    static final String SECURITY_INDEX_TEMPLATE_VERSION_PATTERN = Pattern.quote("${security.template.version}");
    static final Version MIN_READ_VERSION = Version.V_5_0_0;
    private final InternalClient client;
    final AtomicBoolean templateCreationPending = new AtomicBoolean(false);
    final AtomicBoolean updateMappingPending = new AtomicBoolean(false);
    final AtomicReference upgradeDataState = new AtomicReference<UpgradeState>(UpgradeState.NOT_STARTED);
    private final NativeRealmMigrator nativeRealmMigrator;

    public SecurityTemplateService(Settings settings, InternalClient client, NativeRealmMigrator nativeRealmMigrator) {
        super(settings);
        this.client = client;
        this.nativeRealmMigrator = nativeRealmMigrator;
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        if (!event.localNodeMaster()) {
            return;
        }
        ClusterState state = event.state();
        if (state.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            this.logger.debug("template service waiting until state has been recovered");
            return;
        }
        if (!SecurityTemplateService.securityTemplateExistsAndIsUpToDate(state, this.logger)) {
            this.updateSecurityTemplate();
        }
        if (state.metaData().getIndices() != null && !SecurityTemplateService.securityIndexMappingUpToDate(state, this.logger) && this.securityIndexAvailable(state, this.logger)) {
            this.upgradeSecurityData(state, this::updateSecurityMapping);
        }
    }

    private boolean securityIndexAvailable(ClusterState state, Logger logger) {
        IndexRoutingTable routingTable = SecurityTemplateService.getSecurityIndexRoutingTable(state);
        if (routingTable == null) {
            throw new IllegalStateException("Security index does not exist");
        }
        if (!routingTable.allPrimaryShardsActive()) {
            logger.debug("Security index is not yet active");
            return false;
        }
        return true;
    }

    private void updateSecurityTemplate() {
        if (this.templateCreationPending.compareAndSet(false, true)) {
            this.putSecurityTemplate();
        }
    }

    private boolean upgradeSecurityData(ClusterState state, final Runnable andThen) {
        if (this.upgradeDataState.compareAndSet(UpgradeState.NOT_STARTED, UpgradeState.IN_PROGRESS)) {
            final Version previousVersion = SecurityTemplateService.oldestSecurityIndexMappingVersion(state, this.logger);
            this.nativeRealmMigrator.performUpgrade(previousVersion, new ActionListener<Boolean>(){

                @Override
                public void onResponse(Boolean upgraded) {
                    SecurityTemplateService.this.upgradeDataState.set(UpgradeState.COMPLETE);
                    andThen.run();
                }

                @Override
                public void onFailure(Exception e) {
                    SecurityTemplateService.this.upgradeDataState.set(UpgradeState.FAILED);
                    SecurityTemplateService.this.logger.error(() -> new ParameterizedMessage("failed to upgrade security data from version [{}] ", (Object)previousVersion), (Throwable)e);
                }
            });
            return true;
        }
        if (this.upgradeDataState.get() == UpgradeState.COMPLETE) {
            andThen.run();
        }
        return false;
    }

    private void updateSecurityMapping() {
        if (this.updateMappingPending.compareAndSet(false, true)) {
            this.putSecurityMappings();
        }
    }

    private void putSecurityMappings() {
        Map<String, Object> typeMappingMap;
        String template = TemplateUtils.loadTemplate("/security-index-template.json", Version.CURRENT.toString(), SECURITY_INDEX_TEMPLATE_VERSION_PATTERN);
        try {
            typeMappingMap = XContentHelper.convertToMap((XContent)JsonXContent.jsonXContent, template, false);
        }
        catch (ElasticsearchParseException e) {
            this.updateMappingPending.set(false);
            this.logger.error("failed to parse the security index template", (Throwable)e);
            throw new ElasticsearchException("failed to parse the security index template", (Throwable)e, new Object[0]);
        }
        ConcurrentMap<String, PutMappingResponse> updateResults = ConcurrentCollections.newConcurrentMap();
        Map typeMappings = (Map)typeMappingMap.get("mappings");
        int expectedResults = typeMappings.size();
        for (String type : typeMappings.keySet()) {
            Map typeMapping = (Map)typeMappings.get(type);
            this.putSecurityMapping(updateResults, expectedResults, type, typeMapping);
        }
    }

    private void putSecurityMapping(final Map<String, PutMappingResponse> updateResults, final int expectedResults, final String type, Map<String, Object> typeMapping) {
        this.logger.debug("updating mapping of the security index for type [{}]", (Object)type);
        PutMappingRequest putMappingRequest = (PutMappingRequest)this.client.admin().indices().preparePutMapping(SECURITY_INDEX_NAME).setSource(typeMapping).setType(type).request();
        this.client.admin().indices().putMapping(putMappingRequest, new ActionListener<PutMappingResponse>(){

            @Override
            public void onResponse(PutMappingResponse putMappingResponse) {
                if (!putMappingResponse.isAcknowledged()) {
                    SecurityTemplateService.this.updateMappingPending.set(false);
                    throw new ElasticsearchException("update mapping for [{}] security index was not acknowledged", type);
                }
                updateResults.put(type, putMappingResponse);
                if (updateResults.size() == expectedResults) {
                    SecurityTemplateService.this.updateMappingPending.set(false);
                }
            }

            @Override
            public void onFailure(Exception e) {
                SecurityTemplateService.this.updateMappingPending.set(false);
                SecurityTemplateService.this.logger.warn(() -> new ParameterizedMessage("failed to update mapping for [{}] on security index", (Object)type), (Throwable)e);
            }
        });
    }

    private void putSecurityTemplate() {
        this.logger.debug("putting the security index template");
        String template = TemplateUtils.loadTemplate("/security-index-template.json", Version.CURRENT.toString(), SECURITY_INDEX_TEMPLATE_VERSION_PATTERN);
        PutIndexTemplateRequest putTemplateRequest = (PutIndexTemplateRequest)this.client.admin().indices().preparePutTemplate(SECURITY_TEMPLATE_NAME).setSource(new BytesArray(template.getBytes(StandardCharsets.UTF_8)), XContentType.JSON).request();
        this.client.admin().indices().putTemplate(putTemplateRequest, new ActionListener<PutIndexTemplateResponse>(){

            @Override
            public void onResponse(PutIndexTemplateResponse putIndexTemplateResponse) {
                SecurityTemplateService.this.templateCreationPending.set(false);
                if (!putIndexTemplateResponse.isAcknowledged()) {
                    throw new ElasticsearchException("put template for security index was not acknowledged", new Object[0]);
                }
            }

            @Override
            public void onFailure(Exception e) {
                SecurityTemplateService.this.templateCreationPending.set(false);
                SecurityTemplateService.this.logger.warn("failed to put security index template", (Throwable)e);
            }
        });
    }

    static boolean securityIndexMappingUpToDate(ClusterState clusterState, Logger logger) {
        return SecurityTemplateService.securityIndexMappingVersionMatches(clusterState, logger, Version.CURRENT::equals);
    }

    static boolean securityIndexMappingVersionMatches(ClusterState clusterState, Logger logger, Predicate<Version> predicate) {
        return SecurityTemplateService.securityIndexMappingVersions(clusterState, logger).stream().allMatch(predicate);
    }

    public static Version oldestSecurityIndexMappingVersion(ClusterState clusterState, Logger logger) {
        Set<Version> versions = SecurityTemplateService.securityIndexMappingVersions(clusterState, logger);
        return versions.stream().min((o1, o2) -> {
            if (o1.before((Version)o2)) {
                return -1;
            }
            if (o1.after((Version)o2)) {
                return 1;
            }
            return 0;
        }).orElse(null);
    }

    private static Set<Version> securityIndexMappingVersions(ClusterState clusterState, Logger logger) {
        HashSet<Version> versions = new HashSet<Version>();
        IndexMetaData indexMetaData = clusterState.metaData().getIndices().get(SECURITY_INDEX_NAME);
        if (indexMetaData != null) {
            for (Object object : indexMetaData.getMappings().values().toArray()) {
                MappingMetaData mappingMetaData = (MappingMetaData)object;
                if (mappingMetaData.type().equals("_default_")) continue;
                versions.add(SecurityTemplateService.readMappingVersion(mappingMetaData, logger));
            }
        }
        return versions;
    }

    private static Version readMappingVersion(MappingMetaData mappingMetaData, Logger logger) {
        try {
            Map meta = (Map)mappingMetaData.sourceAsMap().get("_meta");
            if (meta == null) {
                return Version.V_2_3_0;
            }
            return Version.fromString((String)meta.get(SECURITY_VERSION_STRING));
        }
        catch (IOException e) {
            logger.error("Cannot parse the mapping for security index.", (Throwable)e);
            throw new ElasticsearchException("Cannot parse the mapping for security index.", (Throwable)e, new Object[0]);
        }
    }

    static boolean securityTemplateExistsAndIsUpToDate(ClusterState state, Logger logger) {
        return SecurityTemplateService.securityTemplateExistsAndVersionMatches(state, logger, Version.CURRENT::equals);
    }

    static boolean securityTemplateExistsAndVersionMatches(ClusterState state, Logger logger, Predicate<Version> predicate) {
        IndexTemplateMetaData templateMeta = state.metaData().templates().get(SECURITY_TEMPLATE_NAME);
        if (templateMeta == null) {
            return false;
        }
        ImmutableOpenMap<String, CompressedXContent> mappings = templateMeta.getMappings();
        for (Object typeMapping : mappings.values().toArray()) {
            CompressedXContent typeMappingXContent = (CompressedXContent)typeMapping;
            try {
                Map<String, Object> typeMappingMap = XContentHelper.convertToMap(new BytesArray(typeMappingXContent.uncompressed()), false, XContentType.JSON).v2();
                assert (typeMappingMap.size() == 1);
                String key = typeMappingMap.keySet().iterator().next();
                Map mappingMap = (Map)typeMappingMap.get(key);
                if (SecurityTemplateService.containsCorrectVersion(mappingMap, predicate)) continue;
                return false;
            }
            catch (ElasticsearchParseException e) {
                logger.error("Cannot parse the template for security index.", (Throwable)e);
                throw new IllegalStateException("Cannot parse the template for security index.", e);
            }
        }
        return true;
    }

    private static boolean containsCorrectVersion(Map<String, Object> typeMappingMap, Predicate<Version> predicate) {
        Map meta = (Map)typeMappingMap.get("_meta");
        if (meta == null) {
            return false;
        }
        return predicate.test(Version.fromString((String)meta.get(SECURITY_VERSION_STRING)));
    }

    public static IndexRoutingTable getSecurityIndexRoutingTable(ClusterState clusterState) {
        IndexMetaData metaData = clusterState.metaData().index(SECURITY_INDEX_NAME);
        if (metaData == null) {
            return null;
        }
        return clusterState.routingTable().index(SECURITY_INDEX_NAME);
    }

    public static boolean securityIndexMappingAndTemplateUpToDate(ClusterState clusterState, Logger logger) {
        if (!SecurityTemplateService.securityTemplateExistsAndIsUpToDate(clusterState, logger)) {
            logger.debug("security template [{}] does not exist or is not up to date, so service cannot start", (Object)SECURITY_TEMPLATE_NAME);
            return false;
        }
        if (!SecurityTemplateService.securityIndexMappingUpToDate(clusterState, logger)) {
            logger.debug("mapping for security index not up to date, so service cannot start");
            return false;
        }
        return true;
    }

    public static boolean securityIndexMappingAndTemplateSufficientToRead(ClusterState clusterState, Logger logger) {
        if (!SecurityTemplateService.securityTemplateExistsAndVersionMatches(clusterState, logger, MIN_READ_VERSION::onOrBefore)) {
            logger.debug("security template [{}] does not exist or is not up to date, so service cannot start", (Object)SECURITY_TEMPLATE_NAME);
            return false;
        }
        if (!SecurityTemplateService.securityIndexMappingVersionMatches(clusterState, logger, MIN_READ_VERSION::onOrBefore)) {
            logger.debug("mapping for security index not up to date, so service cannot start");
            return false;
        }
        return true;
    }

    static enum UpgradeState {
        NOT_STARTED,
        IN_PROGRESS,
        COMPLETE,
        FAILED;

    }
}

