package com.twistpair.wave.thinclient.util;

public interface IWtcMemoryStream
{
    /**
     * Resets the buffer, setting the position AND LENGTH back to 0.<br>
     * Only call this if you want to discard the buffer.<br>
     * Call setPosition(0) if you just want to re-seek to the origin.
     */
    void reset();

    /**
     * Releases the buffer memory and calls reset()
     */
    void clear();

    int getCapacity();

    /**
     * Returns the *ENTIRE ALLOCATED* byte buffer, including any used bytes.<br>
     * Only the first getLength() bytes are actually used.
     * @return the non-null byte buffer
     */
    byte[] getBuffer();

    /**
     * @return the current position for the next read *or* write.
     */
    int getPosition();

    /**
     * Sets the position; will grow the buffer if position > length. 
     * @param position the new position
     */
    void setPosition(int position);

    int incPosition(int length);

    /**
     * @return the length of the stream in bytes.
     */
    int getLength();

    /**
     * If the specified value is less than the current length of the stream, 
     * the stream is truncated and positioned to end.
     * Auto-zero's any bytes added to the existing length.
     * @param length
     */
    void setLength(int length);

    int incLength(int amount);

    /**
     * Writes some or all of a byte array into the buffer.
     * @param src non-null byte buffer
     * @param offset the index of the first byte to write
     * @param count the number of bytes to write
     */
    void write(byte[] src, int offset, int count);

    /**
     * Writes a signed 8-bit byte.
     * @param value signed 8-bit byte.
     */
    void writeInt8(byte value);

    /**
     * Writes an unsigned 8-bit byte.
     * @param value unsigned 8-bit byte.
     */
    void writeUInt8(short value);

    /**
     * Writes a signed 16-bit short.
     * @param value signed 16-bit short
     */
    void writeInt16(short value);

    /**
     * Writes an unsigned 16-bit short.
     * @param value unsigned 16-bit short
     */
    void writeUInt16(int value);

    /**
     * Writes a signed 32-bit integer.
     * @param value signed 32-bit integer.
     */
    void writeInt32(long value);

    /**
     * Writes an unsigned 32-bit integer.
     * @param value unsigned 32-bit integer
     */
    void writeUInt32(long value);

    /**
     * Writes a string using the UTF-8 encoding.
     * @param value string to write
     */
    void writeString(String value);

    int read(byte[] dest, int offset, int count);

    /**
     * Reads a signed 8-bit byte at the current {@link #getPosition()}.<br>
     * ADVANCES THE POSITION!
     * @return signed byte value
     */
    byte readInt8();

    /**
     * Reads an unsigned 8-bit byte at the current {@link #getPosition()}.<br>
     * ADVANCES THE POSITION!
     * @return unsigned 1 byte integer value
     */
    short readUInt8();

    /**
     * Reads a signed 2 byte value at the current {@link #getPosition()}.<br>
     * ADVANCES THE POSITION!
     * @return signed 2 byte value
     */
    short readInt16();

    /**
     * Reads an unsigned 2 byte integer at the current {@link #getPosition()}.<br>
     * ADVANCES THE POSITION!
     * @return unsigned 2byte integer value
     */
    int readUInt16();

    /**
     * Reads a signed 4 byte value at the current {@link #getPosition()}.<br>
     * ADVANCES THE POSITION!
     * @return signed 4 byte value
     */
    int readInt32();

    /**
     * Reads an unsigned 4 byte integer at the current {@link #getPosition()}.<br>
     * ADVANCES THE POSITION!
     * @return unsigned 4 byte integer value
     */
    long readUInt32();

    /**
     * Reads a string at the current {@link #getPosition()}, using the UTF-8 encoding.<br>
     * ADVANCES THE POSITION!
     * @return a non-null string
     */
    String readString();

    String toDebugString();
}
